"""Utilities for file IO operations."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import contextlib
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv

# 4. Local modules


def create_component_folder(comp_uuid):
    """Create a folder for a component in the XMS temp "Components" folder.

    Args:
        comp_uuid (:obj:`str`): UUID of the new component

    Returns:
        (:obj:`str`): Path to the new component folder
    """
    temp_comp_dir = os.path.join(XmEnv.xms_environ_temp_directory(), 'Components', comp_uuid)
    os.makedirs(temp_comp_dir)
    return temp_comp_dir


def logging_filename(abs_path):
    """Get the filename for logging (basename if testing).

    Args:
        abs_path (:obj:`str`): Aboslute path to the file to log

    Returns:
        (:obj:`str`): abs_path or base_name of abs_path if testing
    """
    if XmEnv.xms_environ_running_tests() == 'TRUE':
        return os.path.basename(abs_path)
    return abs_path


@contextlib.contextmanager
def use_cwd(new_cwd):
    """Temporarily set new current working directory.

    Args:
        new_cwd: The new current working directory.
    """
    old_cwd = os.getcwd()
    os.chdir(new_cwd)
    try:
        yield
    finally:
        os.chdir(old_cwd)
