"""Tests for MergeGridTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from osgeo import gdal

# 3. Aquaveo modules

# 4. Local modules


def _run_gdal_function_get_dataset(func, source, destination, **kwargs):
    """Runs the given GDAL function and returns the dataset or whatever the function call returns.

    Args:
        func (function): Any GDAL function that takes source and destination files.
        source (str): The GDAL supported source file(s).
        destination (str): The GDAL supported destination file(s).
        **kwargs: GDAL arguments appropriate to the passed function.

    Returns:
        (gdal.Dataset): The GDAL dataset or whatever the function call returns.
    """
    return func(destination, source, **kwargs)


def _run_gdal_function(func, source, destination, **kwargs):
    """Runs the given GDAL function and returns a True/False based on whether the function was successful.

    Args:
        func (function): Any GDAL function that takes source and destination files.
        source (str): The GDAL supported source file(s).
        destination (str): The GDAL supported destination file(s).
        **kwargs: GDAL arguments appropriate to the passed function.

    Returns:
        (bool): Whether the destination dataset was successfully created.
    """
    gdal.SetConfigOption('GTIFF_REPORT_COMPD_CS', 'TRUE')
    output = func(destination, source, **kwargs)
    ret_val = output is not None and output != 0
    # Close any dataset that was created
    output = None
    return ret_val


def gdal_grid(source, destination, **kwargs):
    """Uses gdal.Grid to create a raster from the source and place it in the destination.

    Args:
        source (str): Any OGR supported readable datasource or filename.
        destination (str): The GDAL supported output raster datasource or filename.
        **kwargs: return of gdal.GridOptions(), string or array of strings, or other keyword arguments of
            gdal.GridOptions(). If this is a gdal.GridOptions() object, other keywords are ignored.

    Returns:
        (bool): Whether the destination dataset was successfully created.
    """
    ret_val = False
    grid_ds = gdal.Grid(destination, source, **kwargs)
    if grid_ds is not None:
        grid_ds.FlushCache()
        grid_ds = None
        ret_val = True
    return ret_val


def gdal_rasterize(source, destination, **kwargs):
    """Uses gdal.Rasterize to burn vector geometry from the source and place it in the destination.

    Args:
        source (str): Any OGR supported readable datasource or filename.
        destination (str): The GDAL supported output raster datasource or filename.
        **kwargs: return of gdal.RasterizeOptions(), string or array of strings, or other keyword arguments of
            gdal.RasterizeOptions(). If this is a gdal.RasterizeOptions() object, other keywords are ignored.

    Returns:
        (bool): Whether the destination dataset was successfully created.
    """
    return _run_gdal_function(gdal.Rasterize, source, destination, **kwargs)


def gdal_rasterize_layer(source, destination, bands, burn_values, options=None):
    """Uses gdal.Rasterize to burn vector geometry from the source and place it in the destination.

    Args:
        source (ogr.Layer): An OGR layer to rasterize.
        destination (gdal.Dataset): The GDAL supported output raster datasource.
        bands (list): A list of bands to burn.
        burn_values (list): A list of burn values for each band.
        options (list): A text list of burn options.

    Returns:
        (bool): Whether the destination dataset was successfully created.
    """
    ret_val = False
    if source is not None and destination is not None:
        output = gdal.RasterizeLayer(destination, bands, source, burn_values=burn_values, options=options)
        ret_val = output is not None and output != 0
        # Close any dataset that was created
        output = None
    return ret_val


def gdal_translate_get_dataset(source, destination, **kwargs):
    """Uses gdal.Translate convert a dataset.

    Args:
        source (str): A GDAL raster object or filename.
        destination (str): The GDAL supported filename.
        **kwargs: return of gdal.TranslateOptions(), string or array of strings, or other keyword arguments of
            gdal.TranslateOptions(). If this is a gdal.TranslateOptions() object, other keywords are ignored.

    Returns:
        (gdal.Dataset): The GDAL dataset or whatever the function call returns.
    """
    return _run_gdal_function_get_dataset(gdal.Translate, source, destination, **kwargs)


def gdal_translate(source, destination, **kwargs):
    """Uses gdal.Translate convert a dataset.

    Args:
        source (str): A GDAL raster object or filename.
        destination (str): The GDAL supported filename.
        **kwargs: return of gdal.TranslateOptions(), string or array of strings, or other keyword arguments of
            gdal.TranslateOptions(). If this is a gdal.TranslateOptions() object, other keywords are ignored.

    Returns:
        (bool): Whether the destination dataset was successfully created.
    """
    return _run_gdal_function(gdal.Translate, source, destination, **kwargs)


def gdal_warp(source, destination, **kwargs):
    """Uses gdal.Warp to warp one or more datasets.

    Args:
        source (str): An array of GDAL raster objects or filenames, or a GDAL raster object, or a filename.
        destination (str): The GDAL supported output raster datasource or filename.
        **kwargs: return of gdal.WarpOptions(), string or array of strings, or other keyword arguments of
            gdal.WarpOptions(). If this is a gdal.WarpOptions() object, other keywords are ignored.

    Returns:
        (bool): Whether the destination dataset was successfully created.
    """
    return _run_gdal_function(gdal.Warp, source, destination, **kwargs)


def gdal_warp_get_dataset(source, destination, **kwargs):
    """Uses gdal.Warp to warp one or more datasets.

    Args:
        source (str): An array of GDAL raster objects or filenames, or a GDAL raster object, or a filename.
        destination (str): The GDAL supported output raster datasource or filename.
        **kwargs: return of gdal.WarpOptions(), string or array of strings, or other keyword arguments of
            gdal.WarpOptions(). If this is a gdal.WarpOptions() object, other keywords are ignored.

    Returns:
        (gdal.Dataset): The GDAL dataset or whatever the function call returns.
    """
    return _run_gdal_function_get_dataset(gdal.Warp, source, destination, **kwargs)


def gdal_build_vrt(source, destination, **kwargs):
    """Creates a VRT file from the given source file(s).

    Args:
        source (str): An array of GDAL raster objects or filenames, or a GDAL raster object, or a filename.
        destination (str): The GDAL supported output raster datasource or filename.
        **kwargs: return of gdal.BuildVRTOptions(), string or array of strings, or other keyword arguments of
            gdal.BuildVRTOptions(). If this is a gdal.BuildVRTOptions() object, other keywords are ignored.

    Returns:
        (bool): Whether the destination dataset was successfully created.
    """
    return _run_gdal_function(gdal.BuildVRT, source, destination, **kwargs)
