"""This module tracks the ongoing progress of a Gencade run."""

# 1. Standard Python modules
from datetime import datetime

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.gencade.data.sim_data import SimData


class GencadeTracker:
    """This class tracks the progress of a Gencade run."""
    prog = None
    query = None
    echo_file = None
    echo_pos = 0
    duration = 0.0
    start_time = 0.0
    end_time = 0.0

    def __init__(self):
        """Does nothing except construct the class."""
        pass

    @staticmethod
    def calculate_progress(time):
        """Calculate progress given a timestep.

        Args:
            time (float): The current time in seconds

        Returns:
            (float): Progress as a percent of all timesteps.

        """
        # Cast to float to avoid truncation if int
        elapsed_time = time - GencadeTracker.start_time
        return (elapsed_time.total_seconds() / float(GencadeTracker.duration.total_seconds())) * 100.0

    @staticmethod
    def progress_function():
        """Method called inside the progress loop for computing percent done."""
        if not GencadeTracker.echo_file:
            GencadeTracker.echo_file = GencadeTracker.prog.command_line_output_file

        found_iter = False
        try:
            with open(GencadeTracker.echo_file, 'r') as f:
                f.seek(GencadeTracker.echo_pos)
                echo_line = f.readline()
                while echo_line:
                    if echo_line.endswith('\n') or echo_line.endswith('\r'):
                        GencadeTracker.echo_pos = f.tell()
                        if echo_line.strip().startswith('CALCULATED'):
                            try:
                                echo_vals = echo_line.split('IS')
                                GencadeTracker.current_time = datetime.strptime(echo_vals[1].strip(), "%Y%m%d")
                            except ValueError:
                                pass
                            found_iter = True
                    echo_line = f.readline()
        except Exception:
            pass  # File might not exist yet

        if found_iter:
            percent_done = GencadeTracker.calculate_progress(GencadeTracker.current_time)
            GencadeTracker.query.update_progress_percent(percent_done)

    @staticmethod
    def start_tracking():
        """Entry point for the GenCade progress script."""
        GencadeTracker.query = Query(progress_script=True)
        GencadeTracker.prog = GencadeTracker.query.xms_agent.session.progress_loop

        # Get the simulation hidden component
        sim_uuid = GencadeTracker.query.current_item_uuid()  # Get the simulation uuid
        sim_comp = GencadeTracker.query.item_with_uuid(sim_uuid, model_name='GenCade',
                                                       unique_name='SimComponent')
        data = SimData(sim_comp.main_file)
        GencadeTracker.start_time = datetime.fromisoformat(data.model.attrs['start_date'])
        GencadeTracker.end_time = datetime.fromisoformat(data.model.attrs['end_date'])
        GencadeTracker.duration = GencadeTracker.end_time - GencadeTracker.start_time

        GencadeTracker.prog.set_progress_function(GencadeTracker.progress_function)
        GencadeTracker.prog.start_loop()
