"""The Structure properties dialog."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import QSize
from PySide2.QtWidgets import QComboBox, QDialogButtonBox, QHBoxLayout, QLabel, QLineEdit, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules


class GridPointAttsDlg(XmsDlg):
    """A dialog for editing structure properties."""

    def __init__(self, parent, dlg_data):
        """Initializes the dialog, sets up the ui.

        Args:
            parent (:obj:`QWidget`): Parent window
            dlg_data (:obj:`dict`): data for the dialog
        """
        super().__init__(parent, 'xms.gencade.gui.grid_point_atts_dlg')
        self._help_url = 'https://cirpwiki.info/wiki/GenCade_2.0_Grid_Point_Attributes'
        self._units = dlg_data['units']
        self._dlg_data = dlg_data
        self._widgets = {}
        self.dbl_validator = None

        self._setup_ui()
        self.setMinimumSize(QSize(400, 200))

    def _setup_ui(self):
        """Sets up the widgets for the UI."""
        self.setWindowTitle('GenCade Grid Point Attributes')
        self.dbl_validator = QxDoubleValidator(parent=self, bottom=1.0)
        self._add_main_layout()
        self._add_widgets_to_main_layout()
        self._add_button_box()

        self._on_change_point_type()

    def _add_main_layout(self):
        """Adds the main dialog layout."""
        self._widgets['main_vert_layout'] = QVBoxLayout()
        self.setLayout(self._widgets['main_vert_layout'])

    def _add_widgets_to_main_layout(self):
        """Adds widgets to the main layout."""
        layout = self._widgets['main_vert_layout']

        self._widgets['txt_point_type'] = QLabel('Point type:')
        layout.addWidget(self._widgets['txt_point_type'])
        grid_types = ['Unassigned', 'Begin', 'End', 'Refine']
        self._widgets['cbx_grid_point_type'] = QComboBox()
        self._widgets['cbx_grid_point_type'].addItems(grid_types)
        pt_type = self._dlg_data['point_type']
        idx = 0 if pt_type not in grid_types else grid_types.index(pt_type)
        self._widgets['cbx_grid_point_type'].setCurrentIndex(idx)
        layout.addWidget(self._widgets['cbx_grid_point_type'])
        self._widgets['cbx_grid_point_type'].currentIndexChanged.connect(self._on_change_point_type)

        self._widgets['txt_cell_size'] = QLabel(f'Refined cell size ({self._units}):')
        layout.addWidget(self._widgets['txt_cell_size'])
        self._widgets['edt_cell_size'] = QLineEdit(f'{self._dlg_data["refine_size_const"]}')
        self._widgets['edt_cell_size'].setValidator(self.dbl_validator)
        self._widgets['edt_cell_size'].setVisible(
            self._widgets['cbx_grid_point_type'].currentText() == 'Refine'
        )
        layout.addWidget(self._widgets['edt_cell_size'])

        layout.addStretch()

    def _add_button_box(self):
        """Adds the button box to the bottom of the dialog."""
        self._widgets['btn_box'] = QDialogButtonBox()
        btn_flags = QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Help
        self._widgets['btn_box'].setStandardButtons(btn_flags)
        self._widgets['btn_box'].accepted.connect(self.accept)
        self._widgets['btn_box'].rejected.connect(self.reject)
        self._widgets['btn_box'].helpRequested.connect(self.help_requested)
        self._widgets['btn_horiz_layout'] = QHBoxLayout()
        self._widgets['btn_horiz_layout'].addWidget(self._widgets['btn_box'])
        self._widgets['main_vert_layout'].addLayout(self._widgets['btn_horiz_layout'])

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)

    def _on_change_point_type(self):
        """Change to the point type combobox."""
        idx = self._widgets['cbx_grid_point_type'].currentIndex()
        self._widgets['txt_cell_size'].setVisible(idx == 3)
        self._widgets['edt_cell_size'].setVisible(idx == 3)

    def get_comp_data(self):
        """Returns the options set in the dialog.

        Returns:
            (:obj:`dict{'point_type": '', 'refine_size_const': ###}`)
        """
        return {
            'point_type': self._widgets['cbx_grid_point_type'].currentText(),
            'refine_size_const': float(self._widgets['edt_cell_size'].text())
        }
