"""GmiComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from functools import cached_property
import os
from pathlib import Path
import shutil
from typing import Optional, TypeVar
import warnings

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Menu, MenuItem, Query
from xms.api.tree import tree_util, TreeNode
from xms.components.bases.coverage_component_base import CoverageComponentBase
from xms.core.filesystem import filesystem as io_util
from xms.data_objects.parameters import Component

# 4. Local modules
from xms.gmi.components.utils import duplicate_display_opts  # noqa: F401  Here for backwards compatibility
from xms.gmi.data.coverage_data import CoverageData
from xms.gmi.data.generic_model import GenericModel, Parameter
from xms.gmi.gui.dataset_callback import DatasetRequest

UNINITIALIZED_COMP_ID = -1

# Type aliases
# See https://stackoverflow.com/a/71441339/5666265
C = TypeVar('C', bound=CoverageData)  # Stops PyCharm's type checking from complaining


def get_component_data_object(main_file: str, comp_uuid: str, unique_name: str, name: str = ''):
    """
    Create a `data_object` `Component` to send back to SMS to be built.

    Args:
        main_file: Path to the component main file.
        comp_uuid: UUID of the component.
        unique_name: XML component unique name.
        name: Tree item name of the component.

    Returns:
        Component: data_object for the new component.
    """
    return Component(
        name=name, comp_uuid=comp_uuid, main_file=main_file, model_name='GMI', unique_name=unique_name, locked=False
    )


class GmiComponent(CoverageComponentBase):
    """A Dynamic Model Interface (DMI) component base for the GMI model."""
    def __init__(self, main_file: Path | str, generic_model: GenericModel = None):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
            generic_model: Parameter definitions.
        """
        super().__init__(str(main_file))  # CoverageComponentBase doesn't like Path objects.
        self.__class_name = None
        self.__module_name = None
        self._generic_model = generic_model
        self.tree_commands = []  # [(menu_text, menu_method)...]
        self.point_commands = []  # [(menu_text, menu_method)...]
        self.arc_commands = []  # [(menu_text, menu_method)...]
        self.polygon_commands = []  # [(menu_text, menu_method)...]

        self._project_tree: Optional[TreeNode] = None
        self._query: Optional[Query] = None

        # Some outside code assumes the main file has been created after the
        # constructor returns. It might only be tests that make this
        # assumption, so maybe we can remove it at some point. For now though,
        # we read `self.data` so it will create the data manager and create
        # the main file.
        _ = self.data

    @cached_property
    def data(self) -> C:
        """
        The component's data manager.

        Most derived classes will want their own data manager. GMI classes can override `self._get_data_with_model()`
        to accomplish this. Derived DMIs should override `self._get_data()` instead. See `self._get_data_with_model()`
        for details.
        """
        model = self._generic_model
        del self._generic_model

        if hasattr(self, '_get_data'):
            return self._get_data()
        else:
            return self._get_data_with_model(model)

    @property
    def uuid(self):
        """
        The component's UUID.

        Before GMI, most components would assign this in their `__init__()`. The initialization was always the same,
        every single time, so GMI-based components offload it into the data manager, which has a single implementation
        so everyone else can just ignore it.
        """
        return self.data.uuid

    @uuid.setter
    def uuid(self, _):
        warnings.warn('Setting UUIDs on GMI-based components is ignored.', category=DeprecationWarning, stacklevel=2)

    def _get_data_with_model(self, model: GenericModel):
        """
        Get a data manager for this component.

        This saves a copy of the generic model into the file, which is useful for DMIs (like GMI) that have to deal with
        arbitrary models at runtime and can't make any assumptions about them. It's usually redundant for derived DMIs
        though, since they typically only have a single model that is baked into the code.

        Derived DMIs should typically override `self._get_data()` instead, which will look like this:

        ```
        def _get_data(self):
            return SomeDataManager(self.main_file)
        ```

        GMI classes override this to provide their own data manager. Implementations will typically look something like

        ```
        def _get_data_with_model(self, model: GenericModel):
            return SomeDataManager(self.main_file, model)
        ```
        """
        return CoverageData(self.main_file, model)

    def _get_menu_action(
        self,
        command_method: str,
        id_files: Optional[tuple[str, str]] = None,
        selection: Optional[list[int]] = None,
        main_file: Optional[str] = None
    ):
        """Get an ActionRequest for a modal menu item command.

        Args:
            command_method: Name of the method to call
            id_files: Paths to the XMS id and component id files, if applicable
            selection: Feature ids of the selected items
            main_file: Path to the component main file.

        Returns:
            ActionRequest: The component menu item ActionRequest
        """
        main_file = main_file if main_file is not None else self.main_file
        action = ActionRequest(
            main_file=main_file,
            modality='MODAL',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name=command_method
        )
        parameters = {}
        if id_files is not None:
            parameters['id_files'] = id_files
        if selection is not None:
            parameters['selection'] = selection
        if parameters:
            action.action_parameters = parameters
        return action

    def _get_menu_item(
        self,
        command_text: str,
        command_method: str,
        id_files: Optional[tuple[str, str]] = None,
        selection: Optional[list[int]] = None,
        main_file: Optional[str] = None
    ) -> MenuItem:
        """Get a menu item for a modal command.

        Args:
            command_text: The menu text
            command_method: Name of the method to call
            id_files: Paths to the XMS id and component id files, if applicable
            selection: Feature ids of the selected items
            main_file: Path to the component main file.

        Returns:
            The component menu item
        """
        action = self._get_menu_action(command_method, id_files=id_files, selection=selection, main_file=main_file)
        return MenuItem(text=command_text, action=action)

    def save_to_location(self, new_path: str, save_type: str) -> tuple[str, list[tuple[str, str]], list[ActionRequest]]:
        """Save component files to a new location.

        Args:
            new_path: Path to the new save location.
            save_type: One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                - DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                  start with.
                - PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                  data must use relative file paths.
                - SAVE happens when re-saving this project.
                - SAVE_AS happens when saving a project in a new location. This happens the first time we save a
                  project.
                - UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder
                  in the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            A tuple of (new_main_file, messages, action_requests).
                - new_main_file: Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages: List of tuples with the first element of the tuple being the message level (DEBUG, ERROR,
                  WARNING, INFO) and the second element being the message text.
                - action_requests: List of actions for XMS to perform.
        """
        messages = []
        action_requests = []

        new_main_file = os.path.join(new_path, os.path.basename(self.main_file))

        if save_type == 'SAVE':  # We already updated ourselves, so copy our mainfile from temp to the new location.
            io_util.copyfile(self.main_file, new_main_file)

        return new_main_file, messages, action_requests

    def get_project_explorer_menus(self, main_file_list: list[tuple[str, int]]) -> list[Menu | MenuItem | None]:
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list: List of tuples where the first element is the path to a main_file, and the second
                            element is something else. Not sure what it is, but this ignores it.

        Returns:
            A list of menus and menu items to be shown. Note that this list can have objects of type `xms.api.dmi.Menu`
            as well as `xms.api.dmi.MenuItem`. `None` may be added to the list to indicate a separator.
        """
        if len(main_file_list) > 1 or not main_file_list or not self.tree_commands:
            return []  # Multi-select, nothing selected, or no project explorer menu commands for this component

        menu_list = [None]  # None == spacer
        # Add all the project explorer menus
        for command_text, command_method in self.tree_commands:
            menu_list.append(self._get_menu_item(command_text, command_method, main_file=main_file_list[0][0]))
        return menu_list

    def get_double_click_actions(self, lock_state: bool) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            lock_state: Whether the component is locked for editing. Do not change the files if locked.

        Returns:
            A tuple containing two items.

                - messages: List of tuples with the first element of the tuple being the message level (DEBUG, ERROR,
                  WARNING, INFO) and the second element being the message text.
                - action_requests: List of actions for XMS to perform.
        """
        messages = []
        actions = []

        if self.tree_commands:  # If tree commands have been defined, the first will be the double-click action.
            actions.append(self._get_menu_action(self.tree_commands[0][1]))

        return messages, actions

    def get_display_menus(
        self, selection: dict[str, list[int]], lock_state: bool, id_files: dict[str, tuple[str, str]]
    ) -> list[Menu | MenuItem | None]:
        """
        This will be called when right-click menus in the main display area of XMS are being created.

        Args:
            selection: A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                The value of the dictionary is a list of ids of the selected feature objects.
            lock_state: True if the component is locked for editing. Do not change the files if locked.
            id_files: Key is entity type string, value is tuple of two str where first is the file location of the XMS
                coverage id binary file. Second is file location of the component coverage id binary file. Only
                applicable for coverage selections. File will be deleted after event. Copy if need to persist.

        Returns:
            A list of menus and menu items to be shown. Note that this list can have objects of type `xms.api.dmi.Menu`
                as well as `xms.api.dmi.MenuItem`. `None` may be added to the list to indicate a separator.
        """
        menu_list = [None]  # None == spacer
        # Copy all the id files to a temporary location. XMS will delete them once this method returns.
        temp_dir = os.path.join(os.path.dirname(self.main_file), 'temp')
        os.makedirs(temp_dir, exist_ok=True)

        unpacked_id_files = {}
        for entity, filenames in id_files.items():
            if not os.path.exists(filenames[0]) or not os.path.exists(filenames[1]):
                continue
            temp_xms_file = os.path.join(temp_dir, os.path.basename(filenames[0]))
            temp_comp_file = os.path.join(temp_dir, os.path.basename(filenames[1]))
            io_util.copyfile(filenames[0], temp_xms_file)
            io_util.copyfile(filenames[1], temp_comp_file)
            unpacked_id_files[entity] = (temp_xms_file, temp_comp_file)

        if 'POINT' in selection and selection['POINT']:
            point_id_files = unpacked_id_files['POINT'] if 'POINT' in unpacked_id_files else None
            for command_text, command_method in self.point_commands:
                menu_list.append(
                    self._get_menu_item(
                        command_text, command_method, id_files=point_id_files, selection=selection['POINT']
                    )
                )
        if 'ARC' in selection and selection['ARC']:
            arc_id_files = unpacked_id_files['ARC'] if 'ARC' in unpacked_id_files else None
            for command_text, command_method in self.arc_commands:
                menu_list.append(
                    self._get_menu_item(
                        command_text, command_method, id_files=arc_id_files, selection=selection['ARC']
                    )
                )
        if 'POLYGON' in selection and selection['POLYGON']:
            poly_id_files = unpacked_id_files['POLYGON'] if 'POLYGON' in unpacked_id_files else None
            for command_text, command_method in self.polygon_commands:
                menu_list.append(
                    self._get_menu_item(
                        command_text, command_method, id_files=poly_id_files, selection=selection['POLYGON']
                    )
                )
        if menu_list == [None]:
            shutil.rmtree(temp_dir, ignore_errors=True)  # Delete the id files if no menus were added.
        return menu_list

    def get_double_click_actions_for_selection(
        self, selection: dict[str, list[int]], lock_state: bool, id_files: dict[str, tuple[str, str]]
    ) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        This will be called when a double-click in the main display area of XMS happened.

        Args:
            selection: A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                The value of the dictionary is a list of ids of the selected feature objects.
            lock_state: True if the component is locked for editing. Do not change the files if locked.
            id_files: Key is entity type string, value is tuple of two str where first is the file location of the XMS
                coverage id binary file. Second is file location of the component coverage id binary file. Only
                applicable for coverage selections. File will be deleted after event and should be copied if persistence
                is needed.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        menus = self.get_display_menus(selection, lock_state, id_files)
        # If selected item menu commands have been defined, the first will be the double-click action.
        actions = [menus[1].action_request] if len(menus) >= 2 else []  # The first one is None for a separator.
        return [], actions

    def get_display_options_action(self) -> ActionRequest:
        """Get an ActionRequest that will refresh the XMS display list for components with display."""
        return ActionRequest(
            main_file=self.main_file,
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='get_initial_display_options',
            comp_uuid=self.uuid
        )

    def _dataset_callback(self, request: DatasetRequest, parameter: Parameter) -> str | TreeNode:
        """
        Handle a request for information when picking a dataset.

        This should match `xms.gmi.gui.dataset_callback.DatasetCallback`.

        Notes:
            - `self._query` will be initialized and available before this method is called.
            - If `request` is `DatasetRequest.GetTree`, the returned node will be a clean copy of the entire project
              tree. Derived classes can safely mutate it for filtering purposes.

        Args:
            request: The requested operation.
            parameter: The parameter the request is for. If `request` is `DatasetRequest.GetLabel`, then
                `parameter.value` will be initialized to the UUID of the dataset to get the label for.

        Returns:
            - If `request` is `DatasetRequest.GetLabel`, returns a label to identify the dataset with to the user.
            - If `request` is `DatasetRequest.GetTree`, returns a tree for picking a new dataset for the parameter.
        """
        if self._project_tree is None:
            self._project_tree = self._query.copy_project_tree()
        if request == DatasetRequest.GetLabel:
            return tree_util.build_tree_path(self._project_tree, parameter.value)
        if request == DatasetRequest.GetTree:
            return self._project_tree
        raise AssertionError('Unknown request.')  # pragma: nocover
