"""Miscellaneous functions dealing with data."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.constraint import Grid
from xms.grid.ugrid import UGrid

# 4. Local modules
from xms.gssha.misc.type_aliases import IntArray


def get_on_off_cells(co_grid: Grid, ugrid: UGrid) -> IntArray:
    """Returns the array of model on/off cells."""
    on_off_cells = np.asarray(co_grid.model_on_off_cells)
    if len(on_off_cells) == 0:
        on_off_cells = np.ones(ugrid.cell_count)
    return on_off_cells


def eq_tol(a, b, tol) -> bool:
    """Returns True if A == B within the tolerance.

    See LT_TOL in xms.

    Args:
        a: Left hand side
        b: Right hand side
        tol: The tolerance

    Returns:
        See description.
    """
    return abs(a - b) <= tol


def lt_tol(a, b, tol) -> bool:
    """Returns True if A < B given the tolerance.

    See LT_TOL in xms.

    Args:
        a: Left hand side
        b: Right hand side
        tol: The tolerance

    Returns:
        See description.
    """
    return b - a > tol


def gt_tol(a, b, tol) -> bool:
    """Returns True if A > B given the tolerance.

    See GT_TOL in xms.

    Args:
        a: Left hand side
        b: Right hand side
        tol: The tolerance

    Returns:
        See description.
    """
    return a - b > tol


def lteq_tol(a, b, tol) -> bool:
    """Returns True if A <= B given the tolerance.

    See LT_TOL in xms.

    Args:
        a: Left hand side
        b: Right hand side
        tol: The tolerance

    Returns:
        See description.
    """
    return lt_tol(a, b, tol) or eq_tol(a, b, tol)
