"""Class to store options for a display list's categories."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field
import datetime

# 2. Third party modules

# 3. Aquaveo modules
from xms.testing.tools import new_uuid

# 4. Local modules
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.target_type import TargetType
from xms.guipy.time_format import ISO_DATETIME_FORMAT


@dataclass
class CategoryDisplayOptionList:
    """A class that holds display options for a single category."""
    target_type: TargetType = TargetType.ugrid_cell_center
    categories: list[CategoryDisplayOption] = field(default_factory=list)
    is_ids: bool = True
    uuid: str = field(default_factory=new_uuid)
    comp_uuid: str = ''
    projection: dict = field(default_factory=dict)  # Only applicable for free location draws
    show_targets: bool = False
    logarithmic_scale: bool = False
    target_scale: float = 1.0
    # The observation reference datetime is only stored here because it needs to accessed by the renderer in XMS.
    obs_reftime: datetime.datetime = field(default_factory=lambda: datetime.datetime(1950, 1, 1))
    # Uuid of dset associated with the observations. Like obs_reftime, only stored here because it needs to accessed...
    obs_dset_uuid: str = ''
    # We needed this separate flag for polygons (as opposed to the 'labels_on' attribute of CategoryDisplayOption)
    # because we actually made rendering of polygon labels work in XMS, but we don't want them on most existing
    # polygon coverages. We wanted the default behavior of never drawing labels on polygons to remain intact.
    enable_polygon_labels: bool = False

    def set_projection(self, coord_sys='', coord_zone=-999, vert_units='', horiz_units='', vert_datum='', wkt=''):
        """Convenience method for setting the optional projection dict.

        Note that the projection info is not applicable to id-based draw categories and will be ignored by XMS if
        specified.

        Args:
            coord_sys (str): The coordinate system of the projection. One of: 'LOCAL', 'GEOGRAPHIC',
                'STATEPLANE, or 'UTM'.
            coord_zone (int): Coordinate system zone number. Only applicable if coord_sys is
                'STATEPLANE' or 'UTM'.
            vert_units (str): Vertical units of the projection. One of: 'ARC_DEGREES', 'FEET (INTERNATIONAL)',
                'FEET (U.S. SURVEY)', 'INCHES', 'METERS, 'CENTIMETERS'.
            horiz_units (str): Horizontal units of the projection. One of: 'ARC_DEGREES', 'FEET (INTERNATIONAL)',
                'FEET (U.S. SURVEY)', 'INCHES', 'METERS, 'CENTIMETERS'.
            vert_datum (str): Vertical datum of the projection. One of: 'NAVD88', 'NGVD29'.
            wkt (str): Well-known text of the projection. If a WKT is provided, all other arguments will be
                ignored by XMS. Prefer specifying the WKT when possible.
        """
        if coord_sys:
            self.projection['coord_sys'] = coord_sys
        if coord_zone != -999:
            self.projection['coord_zone'] = coord_zone
        if vert_units:
            self.projection['vert_units'] = vert_units
        if horiz_units:
            self.projection['horiz_units'] = horiz_units
        if vert_datum:
            self.projection['vert_datum'] = vert_datum
        if wkt:
            self.projection['wkt'] = wkt

    def from_dict(self, opt_dict):
        """Populate from a dict.

        Args:
            opt_dict (dict): dict containing the display list category attributes
        """
        if 'target_type' in opt_dict:
            self.target_type = TargetType(int(opt_dict['target_type']))
        if 'is_ids' in opt_dict:
            self.is_ids = bool(opt_dict['is_ids'])
        if 'categories' in opt_dict:
            for cat in opt_dict['categories']:
                opt_cat = CategoryDisplayOption()
                opt_cat.from_dict(cat, self.target_type)
                self.categories.append(opt_cat)
        if 'uuid' in opt_dict:
            self.uuid = opt_dict['uuid']
        if 'comp_uuid' in opt_dict:
            self.comp_uuid = opt_dict['comp_uuid']
        if 'projection' in opt_dict:
            self.projection = opt_dict['projection']
        if 'show_targets' in opt_dict:
            self.show_targets = bool(opt_dict['show_targets'])
        if 'logarithmic_scale' in opt_dict:
            self.logarithmic_scale = bool(opt_dict['logarithmic_scale'])
        if 'target_scale' in opt_dict:
            self.target_scale = float(opt_dict['target_scale'])
        if 'obs_reftime' in opt_dict:
            self.obs_reftime = datetime.datetime.strptime(opt_dict['obs_reftime'], ISO_DATETIME_FORMAT)
        if 'obs_dset_uuid' in opt_dict:
            self.obs_dset_uuid = opt_dict['obs_dset_uuid']
        if 'enable_polygon_labels' in opt_dict:
            self.enable_polygon_labels = bool(opt_dict['enable_polygon_labels'])

    def __iter__(self):
        """For converting to a dict."""
        for cat in self.categories:
            yield dict(cat)
