"""DateTimeDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QDate, QDateTime, Qt, QTime
from PySide2.QtWidgets import QDialog

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.dialogs.date_time_dialog_ui import Ui_DateTimeDialog
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg


class DateTimeDialog(XmsDlg):
    """Dialog to get a date/time from the user."""
    def __init__(self, date: QDate | None = None, time: QTime | None = None, parent=None):
        """Initializes the dialog.

        Args:
            date (QDate|None): The initial date.
            time (QTime|None): The initial time.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent, 'xms.guipy.dialogs.date_time_dialog')

        self.ui = Ui_DateTimeDialog()
        self.ui.setupUi(self)

        if date:
            self.ui.calendar.setSelectedDate(date)
        if time:
            self.ui.time_edit.setTime(time)

    def get_date_time_string(self, format_: Qt.DateFormat = Qt.ISODate):
        """Returns the date/time as a string, like "2023-10-11T00:00:00".

        Args:
            format_ (Qt.DateFormat): Format passed to QDateTime.toString().

        Returns:
            (str): The date/time as a string formatted as Qt.ISODate.
        """
        return get_date_time_string(self.ui.calendar.selectedDate(), self.ui.time_edit.time(), format_)


def run(date: QDate | None, time: QTime | None, parent=None):
    """Runs the DateTimeDialog.

    Args:
        date (QDate): The initial date.
        time (QTime): The initial time.
        parent (Something derived from QWidget): The parent window.

    Returns:
        (tuple): tuple containing:

            (bool): True on OK, or False on Cancel.

            (QDate|None): The date.

            (QTime|None): The time.
    """
    dialog = DateTimeDialog(date, time, parent)
    if dialog.exec() == QDialog.Accepted:
        return True, dialog.ui.calendar.selectedDate(), dialog.ui.time_edit.time()
    else:
        return False, None, None


def get_date_time_string(date: QDate, time: QTime, format_: Qt.DateFormat = Qt.ISODate):
    """Returns the date/time as a string, like "2023-10-11T00:00:00".

    Args:
        date (QDate|None): The initial date.
        time (QTime|None): The initial time.
        format_ (Qt.DateFormat): Format passed to QDateTime.toString().

    Returns:
        (str): The date/time as a string formatted as Qt.ISODate.
    """
    date_time = QDateTime(date, time)
    return date_time.toString(format_)
