"""HelpGetter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import json
from pathlib import Path
import subprocess
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv

# 4. Local modules
from xms.guipy import file_io_util
from xms.guipy.dialogs import help_cache

# Constants
WAIT_SEC = 5  # How long we'll wait for the internet


class HelpGetter:
    """Returns the url associated with the key.

    Uses a background process to create the cache file, if it doesn't exist, so that opening dialogs is not slowed down.
    My first attempt used threads, but I had all kinds of trouble with that, so I switched to a process.
    """
    def __init__(
        self,
        key: str,
        default: str | None = None,
        dialog_help_url: str | None = None,
        max_wait: int = WAIT_SEC,
        test_fetch_time: int = 0
    ):
        """Initializer.

        Args:
            key: The second part of the wiki help line on the above page (after the '|').
            default: The help page to return if the key isn't found. If None, it will be computed from app name.
            dialog_help_url: Wiki page with all help urls, e.g. 'https://www.xmswiki.com/wiki/GMS:GMS_10.5_Dialog_Help'.
                             If None, it will be computed from app name and version.
            max_wait: Max seconds to wait url() function for process to complete before returning default.
            test_fetch_time: When testing, the time we want it to take to fetch the wiki page.
        """
        self._key = key
        self._default = default if default is not None else _default_help_url()
        self._dialog_help_url = dialog_help_url if dialog_help_url is not None else _default_wiki_dialog_help_url()
        self._max_wait = max_wait
        self._test_fetch_time = test_fetch_time
        self._cache_filepath = _cache_filepath()
        self._process = None

        if not self._cache_filepath.is_file() and self._dialog_help_url:
            self._start_process()

    def _start_process(self) -> None:
        """Start a process to run in the background to fetch the wiki page and create the cache file."""
        flags = subprocess.CREATE_NO_WINDOW | subprocess.DETACHED_PROCESS
        script_filepath = help_cache.script_filepath()
        args = [sys.executable, script_filepath, str(self._cache_filepath), self._dialog_help_url]
        if self._test_fetch_time > 0:
            args.append(str(self._test_fetch_time))
        self._process = subprocess.Popen(args, creationflags=flags)

    def url(self) -> str:
        """Return the url associated with the key."""
        if not self._key:
            return self._default
        elif not self._cache_filepath.is_file() and self._process:
            # Wait up to self._wait_sec for the process to finish. If it still hasn't finished, return the default.
            try:
                self._process.wait(self._max_wait)
            except subprocess.TimeoutExpired:
                return self._default

        # Should be a cache file now
        url = _read_from_cache(self._key, self._cache_filepath)
        if not url:
            _log_read_cache_error(self._key, self._default, self._dialog_help_url)
            return self._default
        return f'https://www.xmswiki.com{url}'


def _log_read_cache_error(key: str, default: str, dialog_help_url: str) -> None:
    """Add an error message to the log file.

    Args:
        key: The second part of the wiki help line on the above page (after the '|').
        default: The help page to return if the key isn't found.
        dialog_help_url: Wiki page with all help urls, e.g. 'https://www.xmswiki.com/wiki/GMS:GMS_10.5_Dialog_Help'.
    """
    with open(XmEnv.xms_environ_debug_file(), 'a') as f:
        d = {
            'message': 'HelpGetter failed to find a wiki page to go to and fell back to the provided default.',
            'info': {
                'key': key,
                'default': default,
                'wiki_dialog_help_url': dialog_help_url
            }
        }
        f.write(json.dumps(d))


def _read_from_cache(key: str, cache_filepath: Path) -> str:
    """Returns the help_url in the cache file in the directory of main_file, or ''.

    Args:
        key: The second part of the wiki help line on the above page (after the '|').

    Returns:
        See description.
    """
    the_dict = file_io_util.read_json_file(cache_filepath)
    return the_dict.get(f'{key}', '')


def _cache_filepath() -> Path:
    """Returns the filepath to the help url cache.

    Returns:
        See description.
    """
    return Path(XmEnv.xms_environ_temp_directory()) / 'help_url_cache.json'


def _default_wiki_dialog_help_url() -> str:
    """Return the URL of the wiki dialog help (e.g. https://www.xmswiki.com/wiki/GMS:GMS_10.5_Dialog_Help)."""
    app_name, app_version = XmEnv.xms_environ_app_name(), XmEnv.xms_environ_app_version()
    if not app_name or not app_version:  # Must be testing. Set to something valid for testing purposes.
        app_name, app_version = 'GMS', '10.5'
    return f'https://www.xmswiki.com/wiki/{app_name}:{app_name}_{app_version}_Dialog_Help'


def _default_help_url() -> str:
    """Return a wiki url that can be used if we have nothing better."""
    app_name = XmEnv.xms_environ_app_name()
    if not app_name:  # Must be testing. Set to something valid for testing purposes.
        app_name = 'GMS'
    return f'https://www.xmswiki.com/wiki/{app_name}:{app_name}'
