"""Dialog for viewing and editing legend display options in XMS."""

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import QSize, Qt, Slot
from PySide2.QtGui import QColor
from PySide2.QtWidgets import QDialog

# 3. Aquaveo modules
from xms.guipy.data.legend_display_options import LegendLocation, LegendOptions
from xms.guipy.dialogs.dialog_util import smart_float
from xms.guipy.dialogs.legend_display_options_ui import Ui_LegendDisplayOptionsDialog
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator

# 4. Local modules


class LegendDisplayOptionsDialog(QDialog):
    """A dialog to let the user set legend display options."""
    def __init__(self, options, parent=None):
        """Initializes the dialog.

        Args:
            options (LegendOptions): The title, color, font, size, and location of the legend to draw.
            parent (Something derived from QWidget): The parent window.

        """
        super().__init__(parent)
        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)

        self.help_url = "https://www.xmswiki.com/wiki/SMS:Materials_Data"

        self.ui = Ui_LegendDisplayOptionsDialog()
        self.ui.setupUi(self)
        self.ui.btn_color.set_color(options.color)
        self.ui.txt_title.setText(options.title)
        if options.is_contour:
            self.ui.tog_equal_height.setChecked(options.equal_color_segment_height)
            self.ui.tog_scientific.setChecked(options.use_scientific)
        else:
            self.ui.tog_equal_height.hide()
            self.ui.tog_scientific.hide()
        self.ui.btn_font.set_color(options.color)
        self.ui.btn_font.set_font(options.font)
        self.ui.edt_height.setText(str(options.size.height()))
        self.ui.edt_width.setText(str(options.size.width()))
        loc_idx = int(options.location)
        self.ui.cbx_location.setCurrentIndex(loc_idx)
        self.ui.edt_x.setText(str(options.x))
        self.ui.edt_y.setText(str(options.y))
        self.ui.edt_z.setText(str(options.z))
        self.enabled_location(loc_idx)
        self.ui.btn_color.current_color_changed[QColor].connect(self.ui.btn_font.set_color)
        self.ui.cbx_location.currentIndexChanged[int].connect(self.enabled_location)
        self.ui.button_box.helpRequested.connect(self.help_requested)

        self.dbl_valid = QxDoubleValidator(parent=self, decimals=8)
        self.int_valid = QxIntValidator(parent=self)
        self.ui.edt_x.setValidator(self.dbl_valid)
        self.ui.edt_y.setValidator(self.dbl_valid)
        self.ui.edt_z.setValidator(self.dbl_valid)
        self.ui.edt_height.setValidator(self.int_valid)
        self.ui.edt_width.setValidator(self.int_valid)

    def get_legend_options(self):
        """Returns the current legend display options.

        Returns:
            options (LegendOptions): The current legend display options.
        """
        options = LegendOptions()
        options.color = self.ui.btn_color.color
        options.title = self.ui.txt_title.text()
        if self.ui.tog_equal_height.isVisible():
            options.equal_color_segment_height = self.ui.tog_equal_height.checked()
            options.use_scientific = self.ui.tog_scientific.checked()
            options.is_contour = True
        else:
            options.is_contour = False
        options.font = self.ui.btn_font.font
        options.size = QSize(int(self.ui.edt_width.text()), int(self.ui.edt_height.text()))
        options.location = LegendLocation(self.ui.cbx_location.currentIndex())
        if self.ui.edt_x.isEnabled():
            options.x = smart_float(self.ui.edt_x.text())
            options.y = smart_float(self.ui.edt_y.text())
        if self.ui.edt_z.isEnabled():
            options.z = smart_float(self.ui.edt_z.text())
        return options

    @Slot()
    def enabled_location(self, idx):
        """Called when the location combo-box changes.

        Args:
            idx (int): The new currect index of the location combo-box.
        """
        loc = LegendLocation(idx)
        if loc == LegendLocation.screen_location:
            self.ui.edt_x.setEnabled(True)
            self.ui.edt_y.setEnabled(True)
            self.ui.edt_z.setEnabled(False)
        elif loc == LegendLocation.world_location:
            self.ui.edt_x.setEnabled(True)
            self.ui.edt_y.setEnabled(True)
            self.ui.edt_z.setEnabled(True)
        else:
            self.ui.edt_x.setEnabled(False)
            self.ui.edt_y.setEnabled(False)
            self.ui.edt_z.setEnabled(False)

    @Slot()
    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    @Slot()
    def accept(self):
        """Called when the OK button is clicked."""
        QDialog.accept(self)

    @Slot()
    def reject(self):
        """Called when the Cancel button is clicked."""
        QDialog.reject(self)
