"""Dialog for viewing and editing line display options in XMS."""

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt, Slot
from PySide2.QtWidgets import QDialog

# 3. Aquaveo modules
from xms.guipy.data.line_style import LineOptions, LineStyle
from xms.guipy.dialogs.line_display_options_ui import Ui_LineDisplayOptionsDialog

# 4. Local modules


class LineDisplayOptionsDialog(QDialog):
    """A dialog to let the user set line display options."""
    def __init__(self, options, parent=None, change_color=False):
        """Initializes the dialog.

        Args:
            options (LineOptions): The color, width, and style of the line to draw.
            parent (Something derived from QWidget): The parent window.
            change_color (bool): If True, the user can change the line color
        """
        super().__init__(parent)
        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)

        self.help_url = "https://www.xmswiki.com/wiki/SMS:Display_Options"

        self.ui = Ui_LineDisplayOptionsDialog()
        self.ui.setupUi(self)
        if change_color:
            self.ui.color_btn.set_color(options.color)
        else:  # Hide color selector widgeta if disabled.
            self.ui.color_btn.hide()
            self.ui.lab_line_color.hide()
        self.ui.spin_width.setValue(options.width)
        if options.style == LineStyle.SOLIDLINE:
            self.ui.btn_solid.setChecked(True)
        elif options.style == LineStyle.DASHEDLINE:
            self.ui.btn_dashed.setChecked(True)
        self.end_point = options.end_point
        self.vertex = options.vertex
        self.point_style = options.point_style

    def get_options(self):
        """Returns the current display options.

        Returns:
            options (LineOptions): The current display options.
        """
        options = LineOptions()
        options.color = self.get_color()
        options.width = self.get_width()
        options.style = self.get_style()
        options.end_point = self.end_point
        options.vertex = self.vertex
        options.point_style = self.point_style
        return options

    def get_color(self):
        """Returns the current color.

        Returns:
            color (QColor): The current color.
        """
        return self.ui.color_btn.color

    def get_width(self):
        """Returns the current width.

        Returns:
            width (int): The current width.
        """
        return self.ui.spin_width.value()

    def get_style(self):
        """Returns the current line style.

        Returns:
            style (LineStyle): The current line style.
        """
        style = LineStyle.SOLIDLINE
        if self.ui.btn_dashed.isChecked():
            style = LineStyle.DASHEDLINE
        return style

    @Slot()
    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    @Slot()
    def accept(self):
        """Called when the OK button is clicked."""
        QDialog.accept(self)

    @Slot()
    def reject(self):
        """Called when the Cancel button is clicked."""
        QDialog.reject(self)
