"""Dialog for printing feedback to the user during long-running operations."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import logging
import os
from pathlib import Path
import sys
from typing import Optional
import warnings

# 2. Third party modules
from PySide2.QtCore import QCoreApplication, QObject, Qt, QThread, Signal
from PySide2.QtGui import QColor, QFont, QMovie
from PySide2.QtWidgets import QDialog, QDialogButtonBox, QWidget
from testfixtures import LogCapture

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv

# 4. Local modules
from xms.guipy.dialogs import windows_gui, xms_parent_dlg
from xms.guipy.dialogs.dialog_util import ensure_qapplication_exists
from xms.guipy.dialogs.feedback_thread import FeedbackThread
from xms.guipy.dialogs.process_feedback_dlg_ui import Ui_ProcessFeedbackDlg
from xms.guipy.dialogs.process_feedback_thread import ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg


def extract_level(msg):
    """Looks for $XMS_LEVEL$ at end of msg, extracts it and returns the level number immediately following it as an int.

    Args:
        msg:

    Returns:
        (int): The log level.
    """
    pos = msg.find(LogEchoQtHandler.xms_level_string)
    if pos > -1:
        level = int(msg[pos + len(LogEchoQtHandler.xms_level_string):])
        return msg[:pos], level
    else:
        return msg, 20  # This shouldn't happen


class LogEchoQtHandler(logging.Handler):
    """Handler for redirecting logging module messages to dialog."""

    xms_level_string = '$XMS_LEVEL$'
    log_level_critical = 50
    log_level_error = 40
    log_level_warning = 30

    def __init__(self):
        """Construct the handler."""
        super().__init__()

    def emit(self, record):
        """Output a message."""
        levelno = record.levelno
        if record.exc_info is not None:
            # Echo traceback to 'python_debug.log' file in the XMS temp directory.
            XmEnv.report_error(record.exc_info[1], log_file=XmEnv.xms_environ_debug_file())
            record = str(record.exc_info[1])  # But only report the actual error message to the user.
        else:  # No exception info, just use default formatting for the log message.
            record = self.format(record)
        if record:
            # Embed the log level in the message. We did this after trying a few other approaches which didn't work
            # (like using a class global, defining a set_level() method, adding an arg to the write() method.
            LogEchoQSignalStream.stdout().write('%s%s%d' % (record, LogEchoQtHandler.xms_level_string, levelno))


class LogEchoQSignalStream(QObject):
    """Dummy stream for firing off echo QSignals when logging message is added."""
    _stdout = None
    _stderr = None
    logged_error = False  # Red - very bad
    logged_warning = False  # Yellow - kind of bad
    message_logged = Signal(str, int)
    blocking_message_logged = Signal(str, int)

    def __init__(self):
        """Constuct the stream."""
        super().__init__()

    @staticmethod
    def reset_flags():
        """Reset the error and warning flags to successful state."""
        LogEchoQSignalStream.logged_error = False
        LogEchoQSignalStream.logged_warning = False

    @staticmethod
    def stdout():
        """Redirect stdout."""
        if not LogEchoQSignalStream._stdout:
            LogEchoQSignalStream._stdout = LogEchoQSignalStream()
            sys.stdout = LogEchoQSignalStream._stdout
        return LogEchoQSignalStream._stdout

    @staticmethod
    def stderr():
        """Redirect stderr."""
        if not LogEchoQSignalStream._stderr:
            LogEchoQSignalStream._stderr = LogEchoQSignalStream()
            sys.stderr = LogEchoQSignalStream._stderr
        return LogEchoQSignalStream._stderr

    def write(self, log_message):
        """Fire off a signal so the log message can be echoed.

        Args:
            log_message (str): str object containing the text to write.

        """
        if not self.signalsBlocked():
            log_message, level = extract_level(log_message)

            # If we change the format of the log messages, make sure to update this logic for detecting fatal errors.
            if level >= LogEchoQtHandler.log_level_error:
                LogEchoQSignalStream.logged_error = True
            elif level == LogEchoQtHandler.log_level_warning:
                LogEchoQSignalStream.logged_warning = True
            if ensure_qapplication_exists().thread() != QThread.currentThread():
                self.blocking_message_logged.emit(log_message, level)
            else:
                self.message_logged.emit(log_message, level)

    def flush(self):
        """Do nothing implementation."""
        pass

    def fileno(self):
        """Do nothing implementation."""
        return -1


class ProcessFeedbackDlg(XmsDlg):
    """
    A dialog for viewing GUI feedback during a long-running process.

    Using this requires some special boilerplate. See `run_feedback_dialog()` below for an alternative that doesn't.
    """
    def __init__(self, display_text, logger_name=None, worker=None, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            display_text (dict): Text to be displayed in the GUI::

                {
                    'title': 'The dialog window title',
                    'working_prompt': 'Initial "Please wait..." message in prompt label',
                    'error_prompt': 'Prompt message to display when ERROR or CRITICAL message logged.',
                    'warning_prompt': 'Prompt message to display when WARNING message logged.',
                    'success_prompt': 'Prompt message to display when processing completes successfully.',
                    'note': 'Text in operation specific additional notes label',
                    'auto_load': 'Text for the autoload toggle (automatically close the dialog). If empty no toggle.',
                    'log_format': '%(levelname)-8s - %(asctime)s - %(name)s - %(message)s',
                    'date_format': '%Y-%m-%d %H:%M:%S'
                    'use_colors': False,  # If True, warnings are green, errors are red and don't have ******
                }

            logger_name (str): Ignored and deprecated. Will be removed eventually. Users are encouraged to switch to
                passing worker and parent parameters by keyword so they won't break when this gets removed.
            worker (QThread): The processing worker thread. Needs to have a signal named 'processing_finished'
                that is emitted whenever the worker thread finishes its stuff. If errors occur in the worker thread, use
                the logging module to log them at ERROR or CRITICAL level. User will be notified and prompted to check
                the log output window. Catch your exceptions in the worker thread and use the logging module to log
                an ERROR or CRITICAL level message. Use logger's exception convenience method inside except clauses to
                log an ERROR level message with a nice traceback for debuggin. DEBUG, INFO, and WARNING level messages
                are echoed to the dialog's log output window, but they are not presented to the user as fatal to the
                processing operation.
            parent (Something derived from QWidget): The parent window

        """
        super().__init__(parent, 'xmsguipy.dialogs.process_feedback_dlg')

        if logger_name is not None:
            warnings.warn(
                'The logger_name parameter is ignored and will be removed eventually.',
                DeprecationWarning,
                stacklevel=2
            )

        self.ui = Ui_ProcessFeedbackDlg()
        self.ui.setupUi(self)
        self.display_text = display_text
        self.finished = False
        self.indicator = None
        self.setWindowTitle(self.display_text.get('title', 'Processing...'))
        self.worker = worker
        self.timer = None
        self.start_time = None
        self.testing = False  # Don't hang dialog if testing.
        self.log_capture = None
        self.handler = None

        # Set up the logging listener to fire off signals whenever logging module messages are logged so they can
        # be echoed to the dialog's log output window.
        # This used to use logger_name, but that results in filtering out messages from other packages (e.g. if HydroAS
        # passed xms.hydroas, then none of the messages from xms.snap would be logged). We always capture everything
        # under xms now so that all our packages' messages get logged.
        self.logger = logging.getLogger('xms')
        self.logger.setLevel(logging.INFO)
        self.handler = LogEchoQtHandler()

        # Set format
        fmt = self.display_text.get('log_format', '%(levelname)s - %(asctime)s - %(name)s - %(message)s')
        datefmt = self.display_text.get('date_format', '')
        formatter = logging.Formatter(fmt, datefmt)
        formatter.default_msec_format = '%s.%03d'
        self.handler.setFormatter(formatter)
        self.ui.txt_log.setTabStopDistance(60)
        # Use a monospace font so we can print pretty tables. This one is not obnoxious and should always be installed.
        # I didn't make this a kwarg because it seems we should be consistent here.
        self.ui.txt_log.setFont(QFont('Courier'))

        self.logger.addHandler(self.handler)

        self._setup_ui()

    def _setup_ui(self):
        """Add the programmatic widgets."""
        # Disable "X" close button
        self.setWindowFlags(self.windowFlags() & ~Qt.WindowCloseButtonHint)

        # Populate label text
        self.ui.lbl_status.setText(self.display_text.get('working_prompt', ''))
        self.ui.lbl_note.setText(self.display_text.get('note', ''))
        # Set text for autoload toggle, hide if not present.
        if 'auto_load' in self.display_text and self.display_text['auto_load']:
            self.ui.tog_auto_close.setText(self.display_text['auto_load'])
        else:
            self.ui.tog_auto_close.setCheckState(Qt.Unchecked)
            self.ui.tog_auto_close.setVisible(False)

        # Connect signal for echoing log messages
        LogEchoQSignalStream.stdout().blocking_message_logged.connect(
            self.on_message_logged, type=Qt.BlockingQueuedConnection
        )
        LogEchoQSignalStream.stderr().blocking_message_logged.connect(
            self.on_message_logged, type=Qt.BlockingQueuedConnection
        )
        LogEchoQSignalStream.stdout().message_logged.connect(self.on_message_logged)
        LogEchoQSignalStream.stderr().message_logged.connect(self.on_message_logged)
        # Disable OK button until we are finished mapping
        self.ui.btn_box.button(QDialogButtonBox.Ok).setEnabled(False)
        # Connect to the finished signal of the worker thread
        self.worker.processing_finished.connect(self.processing_finished)
        # Add the busy indicator label
        self.indicator = QMovie(':resources/animations/load_indicator_small.gif')
        self.ui.lbl_load_indicator.setMovie(self.indicator)
        self.indicator.start()

    def closeEvent(self, event):  # noqa: N802
        """Ignore close event while still performing operation."""
        self._remove_log_handler()
        if not self.finished:
            event.ignore()
        else:
            super().closeEvent(event)

    def accept(self):
        """Override the accept method."""
        self._remove_log_handler()
        super().accept()

    def reject(self):
        """Set flags to prevent sending data to XMS when user cancels."""
        self.finished = True  # No more work to be done
        LogEchoQSignalStream.logged_error = True  # Let calling code know data should not be sent to XMS
        self.worker.quit()  # Kill the worker
        self._remove_log_handler()
        super().reject()  # Close the dialog.

    def exec(self):
        """Override to start the worker thread before exec and wait on it after."""
        self.start_time = datetime.datetime.now()
        self.show()  # Make sure the dialog is visible and widgets are drawn before starting worker thread and exec
        QCoreApplication.instance().processEvents()
        if self.testing and isinstance(self.worker, ProcessFeedbackThread):
            return self._do_test_run()
        else:
            self.worker.start()
        return super().exec_()

    def _remove_log_handler(self):
        """Remove the handler from the logger."""
        if self.handler:
            self.logger.removeHandler(self.handler)
            self.handler = None

    def _do_test_run(self):
        """Run the dialog in testing mode.

        Returns:
            self.reject() or self.accept()
        """
        # This uses LogCapture to save off a copy of the log so we can look for 'ERROR' in it and decide whether to
        # accept or reject. This is called during tests though, and some tests also use LogCapture to record the log.
        # LogCapture doesn't support recursive logging. Those tests won't find any log output. So we also replay all
        # the captured records outside of the context manager to get them to those outside loggers.
        #
        # An unfortunate consequence is the handler that writes to the log file during query playback tests sees all the
        # records twice, so the file has two copies of the log. The feedback dialog probably gets two copies too, but
        # by the time we replay them the window is about to disappear anyway, and it only happens during tests, so
        # nobody cares.
        #
        # FeedbackDialog below uses an alternative system that only observes the log rather than mutating it. It could
        # be used here too, but I don't know how many tests in outside code rely on this behavior, so I'm leaving it
        # alone for now. Hopefully it will just get deleted if/when FeedbackDialog becomes the norm.
        with LogCapture() as lc:
            my_err = None
            try:
                self.worker.do_work()
            except Exception as e:
                my_err = e
            log = [(r.levelno, r.name, r.msg) for r in lc.records]
            log_txt = f'{lc}'
        for r in log:
            level, name, msg, = r
            logging.getLogger(name).log(level, msg)
        if my_err is not None:
            raise my_err
        if 'ERROR' in log_txt:
            self.reject()
            return False
        self.accept()
        return True

    def logged_error(self):
        """Returns True if an error was logged.

        Returns:
            (bool): see above
        """
        return LogEchoQSignalStream.logged_error

    def on_message_logged(self, msg, level):
        """Periodically process Qt events in GUI thread.

        Args:
            msg (str): The log message.
            level (int): The log level.
        """
        # Apply formatting
        msg, old_color, use_colors = self._format_warnings_and_errors(msg, level)
        bold, msg = self._make_bold_if_necessary(msg)

        self.ui.txt_log.append(msg)
        self.ui.txt_log.repaint()

        # Restore formatting back to normal
        if use_colors and level >= LogEchoQtHandler.log_level_warning:
            self.ui.txt_log.setTextColor(old_color)
        if bold:
            self.ui.txt_log.setFontWeight(QFont.Normal)

    def _make_bold_if_necessary(self, msg):
        """Look for keyword '$XMS_BOLD$' in message and if found, remove it, and set font weight to bold.

        Args:
            msg (str): The log message.

        Returns:
            (tuple): tuple containing:

                bold (bool): Flag indicating if font is now bold.

                msg (str): Modified log message without the '$XMS_BOLD$'
        """
        bold = False
        if '$XMS_BOLD$' in msg:
            msg = msg.replace('$XMS_BOLD$', '')
            self.ui.txt_log.setFontWeight(QFont.Bold)
            bold = True
        return bold, msg

    def _format_warnings_and_errors(self, msg, level):
        """Formats warnings green and errors blue or, if not using colors, wraps them in *****.

        Args:
            msg (str): The log message
            level (int): Log levelno

        Returns:
            (tuple): tuple containing:

                msg (str): Modified log message without the '$XMS_BOLD$'

                old_color (QColor): The original text color.

                use_colors (bool): Flag indicating if the color changed.
        """
        # Color warnings and errors or mark them with '*****'
        use_colors = self.display_text.get('use_colors', True)
        old_color = self.ui.txt_log.textColor()
        if level >= LogEchoQtHandler.log_level_error:
            LogEchoQSignalStream.logged_error = True
            if use_colors:
                self.ui.txt_log.setTextColor(QColor(255, 0, 0))  # Red
            else:
                msg = f'\n*****\n{msg}\n*****\n'
        elif level == LogEchoQtHandler.log_level_warning:
            if use_colors:
                self.ui.txt_log.setTextColor(QColor(255, 127, 39))  # Orange
            else:
                msg = f'\n*****\n{msg}\n*****\n'
        return msg, old_color, use_colors

    def processing_finished(self):
        """Called after mapping operation completes. Closes dialog if auto load option enabled."""
        self.logger.info(f'Elapsed time: {datetime.datetime.now() - self.start_time}\n')
        self.finished = True
        self.indicator.stop()
        self.ui.lbl_load_indicator.setVisible(False)
        self.ui.btn_box.button(QDialogButtonBox.Ok).setEnabled(True)  # Re-enable the 'Ok' button
        errors_or_warnings = LogEchoQSignalStream.logged_error or LogEchoQSignalStream.logged_warning
        ok_to_auto_close = self.testing or not errors_or_warnings
        if ok_to_auto_close:  # No errors or warnings logged, check if auto close is enabled
            if self.ui.tog_auto_close.checkState() == Qt.Checked:
                self._remove_log_handler()
                # Close the dialog so we can send mapped data back to SMS
                self.done(QDialog.Accepted)
            else:
                self.ui.lbl_status.setText(self.display_text.get('success_prompt', 'Success'))
                self.ui.lbl_status.setStyleSheet('QLabel{color: rgb(0, 200, 0);}')
                self.ui.tog_auto_close.setEnabled(False)
        else:  # No auto close if warnings/errors and not testing
            if LogEchoQSignalStream.logged_error:
                self.ui.lbl_status.setText(self.display_text.get('error_prompt', 'Error'))
                self.ui.lbl_status.setStyleSheet('QLabel{color: rgb(255, 0, 0);}')
            else:
                self.ui.lbl_status.setText(self.display_text.get('warning_prompt', 'Warning'))
                self.ui.lbl_status.setStyleSheet('QLabel{color: rgb(255, 127, 39);}')
            self.ui.tog_auto_close.setCheckState(Qt.Unchecked)
            self.ui.tog_auto_close.setEnabled(False)


class _Handler(logging.Handler):
    """Helper class to check if an error was logged."""
    def __init__(self):
        """Initialize the class."""
        super().__init__()
        self.setLevel('ERROR')  # We only want to hear about error and above.
        self.logged_error = False

    def emit(self, record):
        """Emit a log message."""
        self.logged_error = True


class FeedbackDialog(ProcessFeedbackDlg):
    """A dialog that displays feedback while running a worker thread."""
    def __init__(self, worker: FeedbackThread, parent: Optional[QWidget]):
        """
        Initialize the feedback dialog.

        Args:
            worker: The worker to run.
            parent: The parent window.
        """
        super().__init__(display_text=worker.display_text, logger_name=worker.logger_name, worker=worker, parent=parent)
        worker.setParent(self)
        self.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
        self._cancelled = False

    def _do_test_run(self) -> bool:
        """Run the dialog in testing mode.

        Returns:
            Whether the dialog was accepted (True) or rejected (False).
        """
        # Under normal circumstances the window remains open indefinitely waiting for the user to click a button.
        # That's fine for interactive use, but it makes tests hang. Under test, we automatically click a button.
        # To accomplish that, we install a special handler into the root log that watches for any errors to be
        # logged. If errors were logged, the user would normally only be allowed to cancel, so we reject in that case.
        # If no errors were logged though, we accept so that the data gets sent. This mimics what a real user would do
        # in these circumstances.
        #
        # Since the handler is installed to the root log, it gets to see all the messages for every log, so we don't
        # have to figure out which logger the feedback thread decided to log to. We'll get its messages anyway.
        handler = _Handler()
        log = logging.getLogger()  # No name provided, so we get the root log.
        log.addHandler(handler)

        self.worker.do_work()

        log.removeHandler(handler)

        if handler.logged_error:
            self.reject()
            return False
        else:
            self.accept()
            return True

    def reject(self):
        """Set flags to prevent sending data to XMS when user cancels."""
        # The superclass assigns this so that calling code won't send the data on failure, but doing so means that
        # calling code can't tell whether it should report cancellation. I'm not sure if changing that would break
        # anything else, so we'll just sabotage it here and tell users to rely on self.successful for that instead.
        old_value = LogEchoQSignalStream.logged_error
        super().reject()
        LogEchoQSignalStream.logged_error = old_value
        self._cancelled = True

    @property
    def cancelled(self) -> bool:
        """
        Check whether the operation was cancelled.

        The operation was cancelled if the user clicked the Cancel, Close, or X buttons.

        This does not consider whether the dialog failed. It is possible for the dialog to be both failed and cancelled
        at the same time, and it is up to user code to decide how to handle that.
        """
        return self._cancelled

    @property
    def failed(self) -> bool:
        """
        Check whether the operation failed.

        The operation fails if an error was logged, or an unhandled exception escaped the feedback thread.

        This does not consider whether the dialog was cancelled. It is possible for the dialog to be both failed and
        cancelled at the same time, and it is up to user code to decide how to handle that.
        """
        # super().reject() makes this return True for cancellation, but this class sabotaged that, so it should only
        # return True for actual failures.
        return self.logged_error()

    @property
    def successful(self) -> bool:
        """
        Check whether the operation was successful.

        The operation is successful if no errors were logged, no exceptions escaped the feedback thread,
        and the user did not cancel.
        """
        return (not self.cancelled) and (not self.failed)


def run_feedback_dialog(worker: FeedbackThread, parent: Optional[QWidget] = None) -> bool:
    """
    Run a worker thread with a feedback dialog.

    If this is an import script and `worker.successful is True` after the thread completes, then `worker.send()` will
    be called to send data to XMS.

    Args:
        worker: Worker thread for the feedback dialog to run.
        parent: Parent window. Script runners should pass `None`, which indicates the parent should be parsed out of
            the command line arguments. Component event handlers can't use this function directly; they should return an
            `ActionRequest` to forward the event to a callback instead. Callbacks always receive a parent and should
            pass it in.

            Note: Older code, and code based on it, often named the parent `win_cont`. This calls it `parent` based on
            the theory that the term `parent` will be much more familiar to new developers. This parent is exactly the
            same as all the other parents in all the other Qt code and all of its documentation. All the stuff that
            applies to those parents applies to this one too.

    Returns:
        Whether the thread completed successfully.

        Threads are considered to have completed successfully if they exited, did not log any errors, did not crash, and
        were not cancelled.

        The return value is also equivalent to whether any data was sent, since data is only sent on success.
    """
    ensure_qapplication_exists()
    is_script_runner = parent is None

    if XmEnv.xms_environ_running_tests() != 'TRUE' and parent is None:
        parent_hwnd, main_hwnd, _ = xms_parent_dlg.parse_parent_window_command_args()
        qt_parent = xms_parent_dlg.get_parent_window_container(parent_hwnd)
        # Create the timer that keeps our Python dialog in the foreground of XMS.
        _ = windows_gui.create_and_connect_raise_timer(main_hwnd, qt_parent)  # Keep the timer in scope
    else:
        qt_parent = None
        main_hwnd = None

    feedback_dialog = FeedbackDialog(worker, parent)
    feedback_dialog.exec()

    stderr_path = os.environ.get(XmEnv.ENVIRON_XMS_STD_ERR_FILE, '')
    if stderr_path and Path(stderr_path).is_file():
        # If stderr exists, it likely contains a stack trace that would be useful for debugging purposes. Unfortunately,
        # it will likely also be deleted before a developer can see it. We'll append it to the debug log.
        stderr = Path(stderr_path).read_text()
        with open(XmEnv.xms_environ_debug_file(), 'a') as f:
            f.write(stderr)

    if is_script_runner and not worker.is_export and feedback_dialog.successful:
        # Component ActionRequests run inside a standard runner that calls `query.send()` for you at the end. Sending
        # here would result in a duplicate send after this function returns, which exposes users to scary error
        # messages. But import runners are all custom, so you have to remember to call `query.send()` at the end or it
        # will look like XMS just ignored your send entirely and you'll probably waste half a day trying to figure out
        # why. Import runners get an extra send here for consistency with action requests. Export runners don't normally
        # send anything, and it's unclear if sending anyway causes trouble, so this only sends for import scripts.
        worker.send()

    if worker.is_export:
        if feedback_dialog.cancelled:
            XmEnv.report_export_aborted()
        elif feedback_dialog.failed:
            XmEnv.report_export_error()

    # This is a singleton, and the feedback dialog uses its state to decide whether it failed. If a feedback dialog
    # fails and you don't reset this, then all future dialogs will claim to have failed whether they did or not. Users
    # won't notice since we only ever run one dialog per process anyway, but tests routinely run a dozen or more dialogs
    # per process, so we reset it here for their benefit.
    LogEchoQSignalStream.reset_flags()

    if qt_parent is not None:
        windows_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground

    return feedback_dialog.successful
