"""BcCoverageComponent class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Group, Section
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.hgs.components.hgs_coverage_component_base import HgsCoverageComponentBase
from xms.hgs.data import bc_generic_model
from xms.hgs.data.domains import Domains
from xms.hgs.mapping.bc_map_att import BcMapAtt


class BcCoverageComponent(HgsCoverageComponentBase):
    """A component for boundary conditions."""
    def __init__(self, main_file: str) -> None:
        """Initializes the class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)

    def _section(self, target_type: TargetType) -> Section:
        """
        Get a GMI Section from the coverage's generic model based on its `TargetType`.

        Args:
            target_type: `TargetType` for the desired section.

        Returns:
            Section: The section for the feature type.
        """
        return bc_generic_model.create().section_from_target_type(target_type)

    def make_map_att(self, feature, att_type: str, group: Group):
        """Returns an MapAtt.

        Args:
            feature: A point, arc, or polygon.
            att_type (str): Attribute type (e.g. 'Flux nodal')
            group (Group): The BC as a generic model group.

        Returns:
            (Type[MapAtt]): The att item.
        """
        return BcMapAtt(feature, att_type, group, self.data)

    @staticmethod
    def group_for_grok(map_att_list: list[BcMapAtt]) -> dict[str, list[BcMapAtt]]:
        """Returns a dict of the BCs sorted into domains.

        'Potential evapotranspiration' is put in ET to write it separately in grok. It's really in OLF.

        Args:
            map_att_list (list[MapAtt]): List of all BC MapAtt objects.

        Returns:
            (dict[str, list[MapAtt]]): See description.
        """
        sorted_bc_atts: dict[str, list[BcMapAtt]] = {Domains.PM: [], Domains.OLF: [], Domains.ET: []}
        for map_att in map_att_list:
            # 'Potential evapotranspiration' is put in ET to write it separately in grok. It's really in OLF.
            if map_att.att_type == 'Potential evapotranspiration':
                sorted_bc_atts[Domains.ET].append(map_att)
            else:
                sorted_bc_atts[map_att.get_domain()].append(map_att)
        return sorted_bc_atts
