"""SectionBase class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules
from xms.core.filesystem import filesystem

# 4. Local modules
from xms.hgs.file_io.file_data import FileData
from xms.hgs.gui import gui_exchange
from xms.hgs.misc import util

# Constants
INDENT_SPACES = 4  # Indent 4 spaces
LENGTH_REQUIRING_END_COMMENT = 20  # More than this many and you get a comment after the 'end'


class SectionBase:
    """Context manager to make writing different sections and indentation easier."""
    def __init__(
        self,
        file_data: FileData,
        heading: str = '',
        write_end: bool = False,
        end_text: str = '',
        indent: bool = False,
        force=False,
        append_blank=False
    ) -> None:
        """Initializes the SectionBase. Either 'file' or 'section' should not be passed, but not both.

        Args:
            file_data (FileData): file_data object.
            heading (str): A heading comment that appears as the first thing in the section.
            write_end (bool): If true, 'end' will be written at the end of the section.
            end_text (str): Text to be written at the end of the 'end' line (e.g. 'end title', 'end ! material'
            indent (bool): If true, the lines in the section will be indented.
            force (bool): If true, the heading and ending will be written even if there was nothing inbetween
            append_blank (bool): If true, a blank line is added after the section.
        """
        self.file_data = file_data
        self._parent_section = file_data.section_stack[-1] if file_data.section_stack else None
        self._heading = heading
        self._write_end = write_end
        self._end_text = end_text
        self._indent = indent
        self._force = force
        self._append_blank = append_blank

        self._indent_level = 1 if indent else 0
        self._first_write = True
        self.lines: list[str] = []  # Lines to be written to the file

        if indent and self._parent_section:  # Increase indent level to 1 more than incoming section
            self._indent_level = self._parent_section._indent_level + 1
        self.file_data.section_stack.append(self)  # Add ourselves to the section stack

    def __enter__(self):
        """Called when we enter."""
        return self

    def __exit__(self, _type, value, traceback):
        """Called when we exit.

        Dumps whatever we cached to either the file or the previous section.
        """
        self._prepend_heading()
        self._append_end()
        self._append_blank_line()
        if self.file_data.file and not self._parent_section:
            self._write_cache_to_file()
        else:
            self._copy_cache_to_parent_section()
        self.file_data.section_stack.pop()  # Pop ourselves off the stack

    def _copy_cache_to_parent_section(self):
        if self._parent_section:
            self._parent_section.lines.extend(self.lines)

    def _write_cache_to_file(self):
        """Writes the cached string to the file if the string isn't empty."""
        if self.lines or self._force:
            self.file_data.file.writelines(line + '\n' for line in self.lines)

    def _prepend_heading(self):
        """Adds the heading if there is something in the cache."""
        if self._heading and (self.lines or self._force):
            self.lines.insert(0, f'{self._indent_str(-1)}{self._heading}')

        # Make sure there's a blank line between the section and whatever is before it
        #   (I don't think we need this anymore, but I'm not 100% certain so I'm just gonna comment it out for now)
        if self.lines and self._parent_section and self._parent_section.lines and self._parent_section.lines[-1] != '':
            self.lines.insert(0, '')

    def _append_end(self):
        """Write the 'end' command if necessary."""
        if (self.lines and self._write_end) or self._force:
            end_str = f'end {self._end_text}' if self._end_text else 'end'
            self.lines.append(f'{self._indent_str(-1)}{end_str}')

    def _append_blank_line(self):
        """Appends a blank line after the section if the section isn't empty."""
        if self._append_blank and self.lines and self.lines[-1] != '':
            self.lines.append('')

    def _indent_str(self, offset):
        """Returns a string comprised of spaces based on current indent level."""
        return max(self._indent_level + offset, 0) * ' ' * INDENT_SPACES

    def _relative_path(self, filepath: str):
        """Returns the relative path to filepath from the file we are writing to."""
        return filesystem.compute_relative_path(str(Path(self.file_data.file.name).parent), filepath)

    def _write_list_or_table(
        self,
        command: str,
        value,
        indent_list: bool,
        item_count: bool = False,
        path_columns: list[int] | None = None
    ) -> list[str]:
        """Writes the list or the table to a list of strings and returns the list of strings.

        Args:
            command (str): grok command
            value: The list or table (list of lists).
            indent_list (bool): For lists/tables, indents the list/table and appends 'end' line.
            item_count (bool): If true, the table of values is preceded by the number of values and there is no 'end'.
            path_columns (list[int] | None): List of columns containing paths so we know to make them relative paths.

        Returns:
            (list[str]): The lines to be written to the file.
        """
        offset = 1  # Table is indented below its command
        lines = []
        if value and len(value) > 0:
            offset = offset - 1 if not indent_list else offset
            if item_count:
                lines.append(f'{self._indent_str(offset)}{len(value)}')
            if util.is_non_string_sequence(value[0]):  # table
                for row in value:
                    s = f'{self._indent_str(offset)}'
                    for i, column in enumerate(row):
                        if path_columns and i in path_columns:
                            column = self._relative_path(column)
                        s += f' {str(column)}' if i > 0 else f'{str(column)}'
                    lines.append(s)
            else:  # list
                for row in value:
                    if path_columns:
                        row = self._relative_path(row)
                    lines.append(f'{self._indent_str(offset)}{str(row)}')

            if indent_list and not item_count:  # This also means add an 'end' line at the end
                if len(value) > LENGTH_REQUIRING_END_COMMENT:
                    lines.append(f'{self._indent_str(0)}end !{command}')  # writes a comment after the 'end'
                else:
                    lines.append(f'{self._indent_str(0)}end')
        return lines

    def _convert_to_lists_if_table(self, table: bool, value):
        """If value is a string defining a pandas table, returns the table values as lists.

        Args:
            table (bool): True if it's a table.
            value: Could be a string defining a pandas table.
        """
        if table and value and isinstance(value, str):
            df = pd.read_json(value, orient='table')
            return df.values.tolist()  # row-wise lists
        return value

    def write_value(
        self,
        command: str,
        value=None,
        one_line: bool = False,
        colon: bool = False,
        append_blank: bool = False,
        indent_list: bool = True,
        item_count: bool = False,
        path_columns: list[int] | None = None,
        table: bool = False
    ) -> None:
        """Writes a command and its associated value.

        Args:
            command (str): grok command
            value: The value. Can be a list, or even a table.
            one_line (bool): If true, writes the value after the variable, otherwise writes it on the next line.
            colon (bool): Only applies if one_line is True. Separates the command from the value with a colon.
            append_blank (bool): If true, appends a blank line after the value.
            indent_list (bool): For lists/tables, indents the list/table and appends 'end' line.
            item_count (bool): If true, the table of values is preceded by the number of values and there is no 'end'.
            path_columns (list[int] | None): List of columns containing paths so we know to make them relative paths.
            table (bool): True if it's a table.
        """
        value = self._convert_to_lists_if_table(table, value)
        lines = []
        offset = 0
        if one_line and value is not None:
            if path_columns:
                value = self._relative_path(value)
            if colon:
                lines.append(f'{self._indent_str(offset)}{command}: {str(value)}')
            else:
                lines.append(f'{self._indent_str(offset)}{command} {str(value)}')
        else:
            lines.append(f'{self._indent_str(offset)}{command}')
            if value is not None:
                if util.is_non_string_sequence(value):
                    lines.extend(self._write_list_or_table(command, value, indent_list, item_count, path_columns))
                else:
                    if path_columns:
                        value = self._relative_path(value)
                    lines.append(f'{self._indent_str(0)}{str(value)}')

        if append_blank:
            lines.append('')
        self.lines.extend(lines)

    def write_widgets(
        self,
        command: str,
        sub_widget_type=None,
        sub_widget_list=None,
        one_line: bool = False,
        colon: bool = False,
        first_widget: str = 'chk',
        item_count: bool = False,
        path_columns: list[int] | None = None,
        table: bool = False
    ) -> None:
        """Writes a command and its associated value, getting the value from the widget info passed.

        - If it's just a command with no other value (just a checkbox), just pass the command.
        - If it has a single value, pass command and sub_widget_type and not sub_widget_list.
        - If it has multiple values, pass command and sub_widget_list containing full widget names.

        Don't pass both sub_widget_type and sub_widget_list.

        Args:
            command (str): Widget identifier.
            sub_widget_type (str): Type abbreviation: 'edt', 'chk' etc.
            sub_widget_list: List of full sub widget names.
            one_line (bool): If true, writes the value after the variable, otherwise writes it on the next line.
            colon (bool): Only applies if one_line is True. Separates the command from the value with a colon.
            first_widget (str): The first widget type (e.g. 'chk' for check box, 'rbt' for radio button).
            item_count (bool): If true, the table of values is preceded by the number of values and there is no 'end'.
            path_columns (list[int] | None): List of columns containing paths so we know to make them relative paths.
            table (bool): True if it's a table.
        """
        assert (not (sub_widget_type and sub_widget_list))  # Pass one or the other (see docstring above)
        widget_name = gui_exchange.get_widget_name(first_widget, command)
        on = self.file_data.data.get(widget_name, False)
        if on:
            if sub_widget_type or sub_widget_list:
                if sub_widget_type:
                    widget_name = gui_exchange.get_widget_name(sub_widget_type, command)
                    value = self.file_data.data.get(widget_name, '')
                    value = self._convert_to_lists_if_table(table, value)
                    indent_list = util.is_non_string_sequence(value)
                else:
                    value = [str(self.file_data.data.get(sub_widget, '')) for sub_widget in sub_widget_list]
                    indent_list = False
                self.write_value(
                    command,
                    value,
                    one_line,
                    colon,
                    append_blank=True,
                    indent_list=indent_list,
                    item_count=item_count,
                    path_columns=path_columns,
                    table=table
                )
            else:
                if first_widget == 'chk' or (first_widget == 'rbt' and not sub_widget_type and not sub_widget_list):
                    value = None
                else:
                    value = on
                self.write_value(
                    command,
                    value,
                    one_line,
                    colon,
                    append_blank=True,
                    item_count=item_count,
                    path_columns=path_columns,
                    table=table
                )

    def write_string(self, s: str):
        """Writes the string to the cache."""
        self.lines.append(s)
        self._first_write = False


class IndentedSection(SectionBase):
    """A section that is indented."""
    def __init__(self, file_data, heading: str, end_text: str = '', append_blank=False) -> None:
        """Initializes the class."""
        super().__init__(file_data, heading, write_end=True, end_text=end_text, indent=True, append_blank=append_blank)


class FlatSection(SectionBase):
    """A section that is not indented and has no 'end' command."""
    def __init__(self, file_data, heading: str) -> None:
        """Initializes the class."""
        super().__init__(file_data, heading, write_end=False, indent=False)


class FlatWithEndSection(SectionBase):
    """A section that is not indented but has an 'end' command ('end title', 'end grid generation')."""
    def __init__(self, file_data, heading: str, end_text: str) -> None:
        """Initializes the class."""
        super().__init__(file_data, heading, write_end=True, end_text=end_text, indent=False, force=True)
