"""TecplotBlockReaderBase class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from io import TextIOWrapper
import logging
from pathlib import Path
import tempfile

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment
from xms.datasets.dataset_writer import DatasetWriter
from xms.guipy.testing import testing_tools

# 4. Local modules
from xms.hgs.file_io import file_io_util


class TecplotBlockReaderBase:
    """Reads a Tecplot ascii block file and makes datasets.

    I assume the file format doesn't vary from the examples I've seen. If so, this will need to be made smarter.
    """
    def __init__(self, filepath: Path, ugrid_uuid: str, dset_time_units: str) -> None:
        """Initializes the file.

        Args:
            filepath (Path): File path.
            ugrid_uuid (str): UUID of the ugrid.
            dset_time_units (str): Dataset time units.
        """
        self._filepath: Path = filepath
        self._ugrid_uuid = ugrid_uuid
        self._dset_time_units = dset_time_units

        self._offset = 0  # To skip over x, y and z
        self._location = ''  # Location of dataset values ('points' or 'cells')
        self._title_key_words = '',  # Tuple of keywords that indicate the title of the file

        self._activity: list[int] | None = None
        self._file: TextIOWrapper | None = None
        self._line = ''  # The last line we read from the file
        self._title = ''
        self._variable_names: list[str] = []
        self._variable_units: dict[int, str] = {}
        self._dataset_writers: list[DatasetWriter] = []
        self._log = logging.getLogger('xms.hgs')

    def read(self):
        """Reads a Tecplot ascii block file."""
        with self._filepath.open('r') as self._file:
            self._read_title()
            self._read_variables()
            self._variable_units, self._line = file_io_util.read_units(self._file)
            self._init_datasets()
            self._read_zones()
        return self._dataset_writers

    def _read_title(self) -> None:
        """Reads the 'TITLE' line."""
        self._title, self._line = read_title(self._title_key_words, self._file)

    def _read_variables(self) -> None:
        """Reads the VARIABLES line."""
        self._variable_names, self._line = read_variables(self._file)

    def _read_zones(self) -> None:
        """Reads the zones."""
        try:
            while self._line:
                if self._line.lower().startswith('zone'):
                    time, variable_values_list = self._read_zone()
                    self._append_to_datasets(time, variable_values_list)
                else:
                    self._line = next(self._file)  # raises StopIteration when we hit EOF, just like we want
        except StopIteration:  # We've reached the end of the file (expected - not exceptional)
            self._finalize_datasets()

    def _read_zone(self) -> tuple[float, list[list[float]]]:
        """Reads a ZONE block.

        Returns:
            (tuple[float, list[list[float]]): The time, and a 2d list = variable, list of values for the variable.
        """
        raise NotImplementedError()  # pragma no cover - shouldn't ever get here

    def _get_timestep_values(self, variable_values: list[float]) -> list[float]:
        """Returns list of values, size of number of UGrid points/cells, with nodata and in right spots."""
        return variable_values

    def _finalize_datasets(self) -> None:
        """Finalizes the datasets."""
        for dataset_writer in self._dataset_writers:
            dataset_writer.active_timestep = 0
            dataset_writer.appending_finished()

    def _append_to_datasets(self, time: float, variable_values_list: list[list[float]]) -> None:
        """Appends the zone data to the datasets."""
        for i, variable_values in enumerate(variable_values_list):
            timestep_values = self._get_timestep_values(variable_values)
            self._dataset_writers[i].append_timestep(time=time, data=timestep_values, activity=self._activity)

    def _read_variable_values(self, count: int) -> list[list[float]]:
        """Reads the variable values."""
        variable_values_list = []
        for variable_name in self._variable_names[self._offset:]:
            self._line = next(self._file)
            self._line = file_io_util.skip_comments(self._file, self._line)
            values = self._read_values(count)
            if variable_name.lower() == 'head':  # We decided to skip reading BC head data
                continue
            variable_values_list.append(values)
        return variable_values_list

    def _read_values(self, count: int) -> list[float]:
        """Read a list of values, size of count."""
        values: list[float] = []
        while True:
            if self._line is not None and not self._line.startswith('#'):
                line_values = [float(item.strip()) for item in self._line.split()]
                values.extend(line_values)
            if len(values) >= count:
                break
            self._line = next(self._file)
        return values

    def _read_zone_string(self, key_string: str) -> str:
        """Reads a string from the ZONE line."""
        pos = self._line.find(key_string)
        pos1 = pos + len(key_string)
        pos2 = self._line.find(',', pos1)
        return self._line[pos1:pos2].strip()

    def _init_dataset_writer(self, variable_name: str, variable_units: str) -> DatasetWriter:
        """Returns a new DatasetWriter."""
        name = f'{self._title}-{variable_name}'
        h5_file = tempfile.NamedTemporaryFile(
            prefix=name, suffix='.h5', delete=False, dir=XmsEnvironment.xms_environ_temp_directory()
        )
        dataset_writer = DatasetWriter(
            h5_filename=h5_file.name,
            name=name,
            dset_uuid=testing_tools.new_uuid(),
            geom_uuid=self._ugrid_uuid,
            num_components=1,
            time_units=self._dset_time_units,
            units=variable_units,
            use_activity_as_null=True,
            location=self._location,
        )
        return dataset_writer

    def _init_datasets(self) -> None:
        """Initializes the dataset writers."""
        for i, variable_name in enumerate(self._variable_names[self._offset:]):
            if variable_name.lower() == 'head':  # We decided to skip reading BC head data
                continue
            variable_units = self._variable_units.get(i + self._offset + 1, '')
            dataset_writer = self._init_dataset_writer(variable_name, variable_units)
            self._dataset_writers.append(dataset_writer)


def read_title(title_key_words, file) -> tuple[str, str]:
    """Reads the title line and returns the title and the last line read.

    Args:
        title_key_words (Iterable[str]): 'TITLE', e.g., after which the title follows.
        file: The file.

    Returns:
        (tuple[str, str]): See description.
    """
    line, title_key_word = file_io_util.skip_to(file, title_key_words, None)
    pos = line.find('=', len(title_key_word))
    if pos < 0:
        pos = line.find(':', len(title_key_word))
    else:
        pos2 = line.lower().find('boundary condition')
        if pos2 >= 0:
            pos = pos2 + len('boundary condition')
    title = line[pos + 1:].strip(' "\n')
    return title, line


def read_variables(file) -> tuple[list[str], str]:
    """Reads the VARIABLES line and returns the list of variables and the last line read.

    Args:
        file: The file.

    Returns:
        (tuple[str, str]): See description.
    """
    line, _ = file_io_util.skip_to(file, ['VARIABLES'], None)
    pos = line.find('=')
    variables = line[pos + 1:].strip(' \n').split(',')
    variable_names = [variable.strip(' "') for variable in variables]  # remove quotes
    return variable_names, line
