"""DebugControlDialog class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QDialog, QDialogButtonBox

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.hgs.gui.debug_control_dialog_ui import Ui_dlg_debug_control


def run(debug_control_filepath: Path, parent=None):
    """Runs the dialog.

    Args:
        debug_control_filepath (Path): Path to the debug.control file.
        parent: Parent widget.
    """
    dialog = DebugControlDialog(debug_control_filepath, parent)
    return dialog.exec()


class DebugControlDialog(XmsDlg):
    """DebugControlDialog class."""
    def __init__(self, debug_control_filepath: Path, parent=None) -> None:
        """Initializes the class.

        Args:
            debug_control_filepath (Path): Path to the debug.control file.
            parent: Parent widget.
        """
        super().__init__(parent, 'xms.hgs.gui.debug_control_dialog')
        self._debug_control_filepath = debug_control_filepath
        self.ui = Ui_dlg_debug_control()
        self.ui.setupUi(self)
        self.ui.txt_file_saved.setVisible(False)

        # Signals
        self.ui.edt_file.textChanged.connect(self._on_text_changed)
        self.ui.buttonBox.button(QDialogButtonBox.Save).clicked.connect(self._on_btn_save)

        self._read_file_to_text_field(debug_control_filepath)

    def _read_file_to_text_field(self, debug_control_filepath: Path) -> None:
        """Reads the file and adds the text to the text field."""
        with debug_control_filepath.open('r') as file:
            self.ui.edt_file.setPlainText(file.read())

    def _write_file_from_text_field(self):
        """Writes the file from the text in the text field."""
        with self._debug_control_filepath.open('w') as file:
            file.write(self.ui.edt_file.toPlainText())

    def _on_text_changed(self):
        """The text changed."""
        self.ui.txt_file_saved.setVisible(False)

    def _on_btn_save(self):
        """On Save."""
        self._write_file_from_text_field()
        self.ui.txt_file_saved.setVisible(True)

    def exec(self):  # pragma no cover - don't test this
        """If testing, just accept immediately."""
        from xms.api.dmi import XmsEnvironment
        if XmsEnvironment.xms_environ_running_tests() == 'TRUE':  # If testing, just accept immediately.
            self.accept()
            return QDialog.Accepted
        else:
            return super().exec_()
