"""PlotDataDialog class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
from PySide2.QtWidgets import QDialog, QHeaderView

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.hgs.gui import gui_util
from xms.hgs.gui.plot_data_dialog_ui import Ui_plot_data_dialog


def run(parent, xy_series_list) -> None:
    """Runs the dialog.

    Args:
        parent: Parent widget.
        xy_series_list (list[xy_series]): The xy series.
    """
    dialog = PlotDataDialog(parent, xy_series_list)
    if XmsEnvironment.xms_environ_running_tests() == 'TRUE':  # If testing, just accept immediately.
        dialog.show()
        dialog.accept()
    else:
        dialog.exec()  # pragma no cover - can't hit this line if testing


class PlotDataDialog(XmsDlg):
    """Shows xy data in a spreadsheet dialog."""
    def __init__(self, parent, xy_series_list):
        """Initializes the class.

        Args:
            parent: Parent widget.
            xy_series_list (list[xy_series]): The xy series.
        """
        super().__init__(parent, 'xms.hgs.gui.xy_data_dialog')

        self._xy_series_list = xy_series_list

        self.ui = Ui_plot_data_dialog()
        self.ui.setupUi(self)
        self.setWindowTitle('Plot Data')
        widget_builder.style_table_view(self.ui.table)
        self._help_getter = gui_util.help_getter('xms.hgs.gui.plot_data_dialog')

        self._add_data()
        self._make_read_only()
        self._size_table()
        self.ui.buttonBox.helpRequested.connect(self.help_requested)

    def _add_data(self) -> None:
        """Adds data to the table."""
        df = self._data_frame_from_xy_series()
        model = QxPandasTableModel(df)
        self.ui.table.setModel(model)

    def _data_frame_from_xy_series(self) -> pd.DataFrame:
        """Returns a DataFrame from the xy series data.

        Returns:
            (DataFrame): The DataFrame.
        """
        data_columns = []
        column_names = []
        for i, xy_series in enumerate(self._xy_series_list):
            if i == 0:  # Only add the time column once
                column_names.append(xy_series.x_title)
                data_columns.append(list(xy_series.x))
            if len(xy_series.y) == len(data_columns[0]):  # Don't add Y column if size doesn't match
                column_names.append(xy_series.y_title)
                data_columns.append(list(xy_series.y))
        list_of_tuples = list(zip(*data_columns))
        df = pd.DataFrame(list_of_tuples, columns=column_names)
        df.index += 1
        return df

    def _make_read_only(self) -> None:
        """Makes the table read-only."""
        self.ui.table.model().set_read_only_columns({i for i in range(self.ui.table.model().columnCount())})

    def _size_table(self) -> None:
        """Sizes the table and columns."""
        self.ui.table.setMinimumHeight(500)
        self.ui.table.setMinimumWidth(400)
        self.ui.table.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)

    def exec(self):  # pragma no cover - don't test this
        """If testing, just accept immediately."""
        from xms.api.dmi import XmsEnvironment
        if XmsEnvironment.xms_environ_running_tests() == 'TRUE':  # If testing, just accept immediately.
            self.accept()
            return QDialog.Accepted
        else:
            return super().exec_()
