"""MaterialComponent data class for material coverage."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = 'All rights reserved'
__all__ = ['MaterialComponent']

# 1. Standard Python modules
from functools import cached_property
from itertools import count
from pathlib import Path
from typing import Optional

# 2. Third party modules
import pandas as pd
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.components.bases.coverage_component_base import ColAttType
from xms.components.bases.visible_coverage_component_base import MessagesAndRequests
from xms.components.display.display_options_helper import MULTIPLE_TYPES, UNASSIGNED_TYPE
from xms.gmi.data.generic_model import Type
from xms.gmi.gui.material_section_dialog import MaterialSectionDialog
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.hydroas.components.coverage_component import CoverageComponent
from xms.hydroas.data.material_data import MaterialData

_shapefile_mapping_column_types = {
    Type.FLOAT: ColAttType.COL_ATT_DBL,
    Type.INTEGER: ColAttType.COL_ATT_INT,
    Type.TEXT: ColAttType.COL_ATT_STR,
}


class MaterialComponent(CoverageComponent):
    """A hidden Dynamic Model Interface (DMI) component for the HYDRO_AS-2D materials coverage."""
    def __init__(self, main_file: Optional[str | Path] = None):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.tree_commands.append(('Materials...', self.view_materials))
        self.polygon_dlg_title = 'HydroAS Material Properties'

    @cached_property
    def data(self) -> MaterialData:
        """The component's data manager."""
        return MaterialData(self.main_file)

    @property
    def features(self) -> dict[TargetType, list[tuple[str, str]]]:
        """The features this coverage supports."""
        section = self.data.generic_model.material_parameters

        features = [(name, section.group(name).label) for name in section.group_names]
        feature_dict = {TargetType.polygon: features}

        return feature_dict

    def _assign_feature(
        self, parent: QWidget, dialog_name: str, window_title: str, target: TargetType, feature_ids: list[int]
    ) -> MessagesAndRequests:
        """
        Display the Assign feature dialog and persist data if accepted.

        Args:
            parent: Parent widget for any dialog windows created.
            dialog_name: name of the dialog
            window_title: title of window
            target: Point, arc, or polygon.
            feature_ids: IDs of selected features.

        Returns:
            Messages and requests for XMS to handle.
        """
        section = self.data.generic_model.material_parameters
        section.restore_values(self.data.material_values)
        section.deactivate_groups()

        component_id = self.get_comp_id(target, feature_ids[0])
        values = self.data.feature_values(target, component_id)
        section.restore_group_activity(values)

        dlg = MaterialSectionDialog(
            parent=parent,
            section=section,
            is_interior=False,
            dlg_name=dialog_name,
            window_title=window_title,
            enable_unchecked_groups=True,
            get_curve=None,
            add_curve=None,
        )
        if dlg.exec():
            # The only thing that matters for the polygon is which material is active.
            active_group = dlg.section.active_group_name(UNASSIGNED_TYPE, MULTIPLE_TYPES)
            values = section.extract_group_activity()
            component_id = self.data.add_feature(target, values, active_group)

            # If the user changed material values, we need to remember that. But group activity is stored at the
            # polygon level, so we throw it out first.
            dlg.section.deactivate_groups()
            self.data.material_values = dlg.section.extract_values()

            # If the user changed the material list, we need to save it. But values don't matter here, so discard them.
            dlg.section.clear_values()
            model = self.data.generic_model
            model.material_parameters = dlg.section
            self.data.generic_model = model

            self.update_feature_types(target)
            self.assign_feature_ids(target, active_group, feature_ids, component_id)

            self.data.commit()

        return [], []

    def view_materials(self, _query: Query, _params: list[dict], parent: QWidget) -> MessagesAndRequests:
        """
        Display a dialog that shows the materials in the coverage.

        Args:
            _query: Unused.
            _params: Unused.
            parent: Unused.

        Returns:
            Messages and requests.
        """
        section = self.data.generic_model.material_parameters
        section.restore_values(self.data.material_values)

        dlg = MaterialSectionDialog(
            parent=parent,
            section=section,
            is_interior=False,
            dlg_name=f'{self.module_name}.assign_point_dialog',
            window_title=self.polygon_dlg_title,
            enable_unchecked_groups=True,
            get_curve=None,
            add_curve=None,
        )
        if dlg.exec():
            # If the user changed material values, we need to remember that. But group activity is stored at the
            # polygon level, so we throw it out first.
            dlg.section.deactivate_groups()
            self.data.material_values = dlg.section.extract_values()

            # If the user changed the material list, we need to save it. But values don't matter here, so discard them.
            dlg.section.clear_values()
            model = self.data.generic_model
            model.material_parameters = dlg.section
            self.data.generic_model = model

            self.update_feature_types(TargetType.polygon)

            self.data.commit()

        return [], []

    def get_table_def(self, target: TargetType) -> list[tuple[str, ColAttType]]:
        """
        Get the shapefile attribute table definition for a target.

        When a shapefile is mapped to a coverage, XMS calls this method to determine what attributes the coverage
        supports. It then allows the user to select which attributes in the shapefile should be mapped to which
        attributes this method identified.

        Args:
            target: Target type to get supported features for.

        Returns:
            List of tuples of (name, type) where name is the user-visible name of an attribute and type is its type.
        """
        if target != TargetType.polygon:
            return []

        section = self.data.generic_model.material_parameters
        table = [
            # This column will contain the feature ID for each polygon. XMS doesn't expose it to the user, but it has to
            # be here or populate_from_att_tables will have no idea which material should be assigned to a polygon.
            ('ID', ColAttType.COL_ATT_INT),
            ('Material name', ColAttType.COL_ATT_STR),
        ]

        group = section.group('0')

        for parameter_name in group.parameter_names:
            parameter = group.parameter(parameter_name)
            if parameter.parameter_type in _shapefile_mapping_column_types:
                col_name = parameter.label
                col_type = _shapefile_mapping_column_types[parameter.parameter_type]
                table.append((col_name, col_type))

        return table

    def populate_from_att_tables(self, att_dfs: dict[TargetType, pd.DataFrame]) -> MessagesAndRequests:
        """
        Populate attributes from an attribute table written by XMS.

        Args:
            att_dfs: Dictionary of attribute pandas.DataFrame. Key is TargetType enum,
                value is None if not applicable.

        Returns:
            Messages and requests.
        """
        if att_dfs[TargetType.point] is not None or att_dfs[TargetType.arc] is not None:
            return [('ERROR', 'An internal error occurred. Please contact Aquaveo Tech Support.')], []
        if att_dfs[TargetType.polygon] is None:
            return [('ERROR', 'An internal error occurred. Please contact Aquaveo Tech Support.')], []

        table = att_dfs[TargetType.polygon]

        if len(table) == 0:
            messages = [
                ('WARNING', 'No polygons were received for mapping. The coverage may have already had polygons.')
            ]
            return messages, []

        self._populate_materials_from_att_table(table)
        self._populate_polygons_from_att_table(table)
        self.data.commit()

        return [], []

    def _populate_materials_from_att_table(self, table: pd.DataFrame):
        """
        Populate the material list from an attribute table.

        Args:
            table: Attribute table. Must have an 'ID' and 'Material name' column. The 'ID' column will be dropped, then
                the rest of the table will be de-duplicated on the 'Material name' column. Materials will be created
                with names from the 'Material name' column, and the remaining columns will be treated as labels of
                parameters to assign values to.
        """
        material_data = table.drop(columns='ID').drop_duplicates(['Material name'])

        model = self.data.generic_model
        section = model.material_parameters
        used_material_names = set(section.group_names)
        unused_material_name = (str(i) for i in count(start=1) if i not in used_material_names)

        for _, row in material_data.iterrows():
            material_name = next(unused_material_name)
            material_label = row['Material name']
            group = section.duplicate_group('0', material_name, material_label)
            for parameter_name in group.parameter_names:
                parameter = group.parameter(parameter_name)
                if parameter.parameter_type not in _shapefile_mapping_column_types:
                    continue
                value = row[parameter.label]
                if parameter.parameter_type == Type.TEXT or (parameter.low <= value <= parameter.high):
                    parameter.value = value

        # Updating feature types depends on the model having been updated.
        model.material_parameters.deactivate_groups()
        self.data.material_values = model.material_parameters.extract_values()
        self.data.generic_model = model
        self.update_feature_types(TargetType.polygon)

    def _populate_polygons_from_att_table(self, table: pd.DataFrame):
        """
        Initialize polygon data from an attribute table.

        Args:
            table: Dataframe with two columns: 'ID' is feature ID of a polygon, and 'Material name' is the name of the
                material assigned to that polygon.
        """
        label_to_name = {}
        section = self.data.generic_model.material_parameters
        for material_name in section.group_names:
            group = section.group(material_name)
            label_to_name[group.label] = material_name

        polygon_data = table[['ID', 'Material name']]
        for _, row in polygon_data.iterrows():
            label = row['Material name']
            name = label_to_name[label]
            feature_id = row['ID']
            component_id = self.data.add_feature(TargetType.polygon, '', name)
            self.assign_feature_ids(TargetType.polygon, name, [feature_id], component_id)
