"""Class to parse a .2dm file."""

# 1. Standard Python modules
import io
from logging import Logger
from typing import Optional, TextIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.coverage.xy.xy_io import XyReader
from xms.coverage.xy.xy_series import XySeries
from xms.guipy.dialogs.log_timer import Timer

# 4. Local modules
from xms.hydroas.file_io.errors import error, GmiError, Messages as Msg

CARD_INDEX = 0
MIN_FIELDS_PER_LINE = 1
MIN_NONEMPTY_LENGTH = 0
NO_FIELDS_LENGTH = 1
FIRST_FIELD = CARD_INDEX + 1
SECOND_FIELD = FIRST_FIELD + 1

BC_CARD_EXTRA_FIELD_LENGTH = 8
BC_CARD_EXTRA_FIELD_INDEX = 6

BC_VAL_CARD_IGNORE_LENGTH = 4
BC_VAL_CARD_MIN_LENGTH = 6
BC_VAL_CARD_MAX_LENGTH = 7
BC_VAL_CARD_FIRST_VARIABLE_FIELD = CARD_INDEX + 5

BEFONT_CARD_SHORT_LENGTH = 3
BEFONT_CARD_LONG_LENGTH = 16
BEFONT_CARD_SHORT_FONT_INDEX = 2
BEFONT_CARD_LONG_FONT_INDEX = 3

BEGCURVE_CARD_LENGTH = 3
BEGCURVE_CARD_LABEL_INDEX = 1
BEGCURVE_CARD_VERSION_INDEX = 2

DEF_CARD_TYPE_INDEX = 4
DEF_CARD_FIXED_FIELDS = NO_FIELDS_LENGTH + 4
DEF_CARD_FIRST_VARIABLE_FIELD = CARD_INDEX + 5

DEP_CARD_MIN_LENGTH = 10  # card, group, param, type, parent, filler, 2*(value, enabled)
DEP_CARD_FILLER_INDEX = CARD_INDEX + 5
DEP_CARD_FIRST_OPTION_INDEX = CARD_INDEX + 6

GP_VAL_CARD_MIN_LENGTH = 4
GP_VAL_CARD_MAX_LENGTH = 5
GP_VAL_CARD_FIRST_VARIABLE_FIELD = CARD_INDEX + 3

MAT_MULTI_MIN_LENGTH = 2
MAT_MULTI_MAX_LENGTH = 3

MAT_VAL_CARD_MIN_LENGTH = 5
MAT_VAL_CARD_MAX_LENGTH = 6
MAT_VAL_CARD_FIRST_VARIABLE_FIELD = CARD_INDEX + 4

OPTS_CARD_MIN_LENGTH = 4
OPTS_CARD_GROUP_INDEX = 1
OPTS_CARD_PARAM_INDEX = 2
OPTS_CARD_FIRST_OPTION_INDEX = 3

SI_CARD_MIN_LENGTH = 2
SI_CARD_MAX_LENGTH = 3


def parse(file: TextIO, log: Optional[Logger] = None) -> tuple[list, dict[int, XySeries]]:
    """
    Parse a GMI definition from a .2dm file.

    Args:
        file: Open file to parse.

    Returns:
        A tuple of (parsed_cards, parsed_curves).
        Raises GmiError on failure.
    """
    parser = GmiParser(file, log=log)
    parser.parse()
    return parser.cards, parser.curves


class GmiParser:
    """Parses generic model interface data out of a .2dm file."""
    def __init__(self, file: TextIO, log: Optional[Logger] = None):
        """
        Initialize the parser.

        Args:
            file: Open file to parse.
            log: Where to log progress to.
        """
        self._log = log or Logger('xms.gmi')
        self._file = file
        self._line = ''
        self._fields = []
        self._line_number = 0
        self._timer = Timer()
        self.cards = []
        self.curves: dict[int, XySeries] = {}

        self._simple_cards = {}
        self._complex_cards = {
            'mesh2d': self._mesh2d_card,
        }

        self._def_patterns = [
            'iitib',  # bool
            'iitiiii',  # integer
            'iitifff',  # float
            'iitit',  # text
            'iitit',  # options
            'iititt',  # curve
            'iitifffqtt',  # float/curve
        ]

    def parse(self):
        """
        Parse the file.

        Raises GmiError on failure.
        """
        for line in self._file:
            self._line_number += 1
            self._line = line
            keep_going = self._parse_line()
            if not keep_going:
                break
            if self._timer.report_due:
                self._log.info(f'Read {self._line_number} cards...')

    def _parse_line(self):
        """Parse a line of the file."""
        self._fields = _split(self._line_number, self._line)

        if len(self._fields) < MIN_FIELDS_PER_LINE:
            self.cards.append([])
            return True

        card = self._fields[CARD_INDEX]

        if card in self._simple_cards:
            pattern = self._simple_cards[card]
            self._parse_fields(pattern)
            self.cards.append(self._fields)
            return True

        elif card in self._complex_cards:
            handler = self._complex_cards[card]
            result = handler()
            if self._fields:
                self.cards.append(self._fields)
            return result is None or result

        else:
            error(self._line_number, Msg.unknown_card, card.upper())

    def _parse_fields(self, pattern):
        """
        Parse a list of fields into basic data types based on a pattern.

        Ignores the first two fields (line number and card). Raises a ParseError if a field couldn't be parsed.

        Args:
            pattern: Each character describes the type of the field in the same location in fields.
                     b -> boolean
                     f -> float
                     i -> integer
                     q -> optionally quoted string
                     t -> text
        """
        if len(self._fields) - 1 < len(pattern):
            error(self._line_number, Msg.missing_field)
        if len(self._fields) - 1 > len(pattern):
            error(self._line_number, Msg.extra_field)

        for index in range(1, len(self._fields)):
            display_field_number = _card_index_to_field_number(index)
            field = self._fields[index]
            field_type = pattern[index - 1]
            self._fields[index] = parse_field(self._line_number, field, field_type, display_field_number)

    def _bc_card(self):
        """Parse a BC card."""
        if len(self._fields) == BC_CARD_EXTRA_FIELD_LENGTH and self._fields[BC_CARD_EXTRA_FIELD_INDEX] == '-1':
            # Some of our users still have .2dm files written by an old SMS that used to write this.
            self._fields.pop(BC_CARD_EXTRA_FIELD_INDEX)
        self._parse_fields('itiibt')

    def _bc_val_card(self):
        """Parse a BC_VAL card."""
        pattern = 'qiii'

        if len(self._fields) == BC_VAL_CARD_IGNORE_LENGTH:
            # SMS used to write a BC_VAL card with no parameter ID or value fields when a particular boundary
            # condition had nothing set. We'll treat it as a no-op for compatibility.
            self._fields = []
            return

        if len(self._fields) < BC_VAL_CARD_MIN_LENGTH:
            error(self._line_number, Msg.missing_field)
        if len(self._fields) > BC_VAL_CARD_MAX_LENGTH:
            error(self._line_number, Msg.extra_field)

        variable_fields = self._fields[BC_VAL_CARD_FIRST_VARIABLE_FIELD:]
        self._fields = self._fields[:BC_VAL_CARD_FIRST_VARIABLE_FIELD]
        self._parse_fields(pattern)

        self._fields += variable_fields

    def _befont_card(self):
        """Parse a BEFONT card."""
        if len(self._fields) == BEFONT_CARD_SHORT_LENGTH:
            self._parse_fields("ii")
        elif len(self._fields) == BEFONT_CARD_LONG_LENGTH:
            self._parse_fields("iiiiiiiiiiiiiit")
        else:
            error(self._line_number, Msg.wrong_parameter_count)

    def _beg2dmbc_card(self):
        """Parse a BEG2DMBC card."""
        self._complex_cards = {
            'end2dmbc': self._end2dmbc_card,
            'bc_val': self._bc_val_card,
            'gp_val': self._gp_val_card,
            'mat_val': self._mat_val_card,
        }

    def _begcurve_card(self):
        """Parse a BEGCURVE card."""
        fields = self._fields[FIRST_FIELD:]
        if fields != ['Version:', '1']:
            error(self._line_number, Msg.unsupported_curve_version)

        # Normally parse() does this, but we're going to consume the endcurve card and want it to still append it.
        self.cards.append(self._fields)
        self._fields = []

        file = io.StringIO()
        for line in self._file:
            if line.lower().startswith('endcurve'):
                self._fields = ['endcurve']
                break
            file.write(line)
            self._line_number += 1
        file.seek(0, io.SEEK_SET)
        xy_reader = XyReader()
        curves = xy_reader.read_from_text_stream(file)

        self.curves = {curve.series_id: curve for curve in curves}

        self._simple_cards = {}
        self._complex_cards = {}
        return False

    def _begparamdef_card(self):
        """Parse a BEGPARAMDEF card."""
        self._simple_cards = {
            'bc_disp_opts': 'iiiiibiii',
            'bcpgc': 'b',
            'bedisp': 'iiiibbiiiiib',
            'disp_opts': 'qiiiibiii',
            'dy': 'b',
            'gm': 't',
            'gp': 'itb',
            'key': 't',
            'mat': 'it',
            'mat_params': 'ii',
            'td': 'ff',
            'tu': 't',
        }

        self._complex_cards = {
            'bc': self._bc_card,
            'bc_def': self._def_card,
            'bc_dep': self._dep_card,
            'bc_opts': self._opts_card,
            'befont': self._befont_card,
            'gp_def': self._def_card,
            'gp_dep': self._dep_card,
            'gp_opts': self._opts_card,
            'endparamdef': self._endparamdef_card,
            'mat_def': self._def_card,
            'mat_dep': self._dep_card,
            'mat_multi': self._mat_multi,
            'mat_opts': self._opts_card,
            'nume': self._null_card,
            'si': self._si_card,
        }

    def _def_card(self):
        """Parse a *_DEF card."""
        try:
            param_type = int(self._fields[DEF_CARD_TYPE_INDEX])
        except IndexError:
            error(self._line_number, Msg.missing_field)
        except ValueError:
            error(self._line_number, 4, Msg.mistyped_field, 'integer')

        try:
            if param_type < 0:
                raise IndexError()
            pattern = self._def_patterns[param_type]
        except IndexError:
            error(self._line_number, 4, Msg.range_error)

        self._parse_fields(pattern)

    def _dep_card(self):
        """Parse a *_DEP card."""
        if len(self._fields) < DEP_CARD_MIN_LENGTH:
            error(self._line_number, Msg.missing_field)
        elif len(self._fields) % 2 == 1:  # odd
            error(self._line_number, Msg.option_mismatch)

        pattern = 'iittb'
        while len(pattern) < len(self._fields) - NO_FIELDS_LENGTH:
            pattern += 'tb'

        self._parse_fields(pattern)

    def _end2dmbc_card(self):
        """Parse an END2DMBC card."""
        self._simple_cards = {}
        self._complex_cards = {
            'begcurve': self._begcurve_card,
        }

    def _endparamdef_card(self):
        """Parse an ENDPARAMDEF card."""
        self._simple_cards = {}
        self._complex_cards = {
            'beg2dmbc': self._beg2dmbc_card,
        }

    def _gp_val_card(self):
        """Parse a GP_VAL card."""
        pattern = 'ii'

        if len(self._fields) < GP_VAL_CARD_MIN_LENGTH:
            error(self._line_number, Msg.missing_field)
        if len(self._fields) > GP_VAL_CARD_MAX_LENGTH:
            error(self._line_number, Msg.extra_field)

        variable_fields = self._fields[GP_VAL_CARD_FIRST_VARIABLE_FIELD:]
        self._fields = self._fields[:GP_VAL_CARD_FIRST_VARIABLE_FIELD]
        self._parse_fields(pattern)

        self._fields += variable_fields

    def _mat_multi(self):
        """Parse a MAT_MULTI card."""
        if len(self._fields) < MAT_MULTI_MIN_LENGTH:
            error(self._line_number, Msg.missing_field)
        elif len(self._fields) == MAT_MULTI_MIN_LENGTH:
            self._parse_fields('b')
            self._fields.append(0)  # We need a group, but it will be ignored, so just make it 0.
        elif len(self._fields) == MAT_MULTI_MAX_LENGTH:
            self._parse_fields('bi')
        else:
            error(self._line_number, Msg.extra_field)

    def _mat_val_card(self):
        """Parse a MAT_VAL card."""
        pattern = 'iii'

        if len(self._fields) < MAT_VAL_CARD_MIN_LENGTH:
            error(self._line_number, Msg.missing_field)
        if len(self._fields) > MAT_VAL_CARD_MAX_LENGTH:
            error(self._line_number, Msg.extra_field)

        variable_fields = self._fields[MAT_VAL_CARD_FIRST_VARIABLE_FIELD:]
        self._fields = self._fields[:MAT_VAL_CARD_FIRST_VARIABLE_FIELD]
        self._parse_fields(pattern)

        self._fields += variable_fields

    def _mesh2d_card(self):
        """Parse a MESH2D card."""
        if len(self._fields) > NO_FIELDS_LENGTH:
            error(self._line_number, Msg.extra_field)

        self._simple_cards = {
            'e3t': 'iiiii',
            'e4q': 'iiiiii',
            'meshname': 't',
            'nd': 'ifff',
            'num_materials_per_elem': 'i',
        }

        self._complex_cards = {
            'begparamdef': self._begparamdef_card,
            'ns': self._ns_card,
        }

    def _ns_card(self):
        """Parse an NS card."""
        if len(self._fields) <= NO_FIELDS_LENGTH:
            error(self._line_number, Msg.missing_field)

        last = self._fields[-1]
        try:
            parse_field(self._line_number, last, 't', 0)
            has_name = True
        except GmiError:
            has_name = False
            try:
                parse_field(self._line_number, last, 'i', 0)
            except GmiError:
                error(self._line_number, len(self._fields) - 1, Msg.not_integer_or_string)

        number_of_fields = len(self._fields) - NO_FIELDS_LENGTH
        if has_name and len(self._fields) < NO_FIELDS_LENGTH + 3:
            error(self._line_number, Msg.missing_field)
        elif not has_name and len(self._fields) < NO_FIELDS_LENGTH + 2:
            error(self._line_number, Msg.missing_field)

        if has_name:
            pattern = 'i' * (number_of_fields - 1) + 't'
        else:
            pattern = 'i' * number_of_fields

        self._parse_fields(pattern)

    def _null_card(self):
        """Parse a card that we don't care about the syntax of."""
        pass

    def _opts_card(self):
        """Parse an OPTS card."""
        # One of our users has simulations with single-option options. Maybe they're trying to hard-code a parameter
        # or something. Whatever their reasons, they generate money, so we generate support.
        if len(self._fields) < OPTS_CARD_MIN_LENGTH:
            error(self._line_number, Msg.missing_field)

        self._fields[OPTS_CARD_GROUP_INDEX] = parse_field(
            self._line_number, self._fields[OPTS_CARD_GROUP_INDEX], 'i', 1
        )
        self._fields[OPTS_CARD_PARAM_INDEX] = parse_field(
            self._line_number, self._fields[OPTS_CARD_PARAM_INDEX], 'i', 2
        )

        for field_number in range(OPTS_CARD_FIRST_OPTION_INDEX, len(self._fields)):
            self._fields[field_number] = parse_field(
                self._line_number, self._fields[field_number], 't', _card_index_to_field_number(field_number)
            )

    def _si_card(self):
        """Parse an SI card."""
        if len(self._fields) < SI_CARD_MIN_LENGTH:
            error(self._line_number, Msg.missing_field)
        elif len(self._fields) == SI_CARD_MIN_LENGTH:
            self._parse_fields("i")
        elif len(self._fields) == SI_CARD_MAX_LENGTH:
            self._parse_fields("it")
        else:
            error(self._line_number, Msg.extra_field)


def parse_field(line_number, field, field_type, field_number):
    """
    Parse a field into a basic data type based on a pattern.

    Raises a ParseError if the field couldn't be parsed.

    Args:
        line_number: Number of the line being parsed. Used for reporting errors.
        field: The field to parse.
        field_type: A character representing the field's expected type.
                    b -> boolean
                    f -> float
                    i -> integer
                    q -> optionally quoted string
                    t -> text
        field_number: The field's number. Used for reporting errors.

    Returns:
        The parsed field.
    """
    if field_type == 'b':
        if field != '0' and field != '1':
            error(
                line_number,
                field_number,
                Msg.mistyped_field,
                'boolean',
            )
        else:
            return True if field == '1' else False
    elif field_type == 'f':
        try:
            return float(field)
        except ValueError:
            error(line_number, field_number, Msg.mistyped_field, 'float')
    elif field_type == 'i':
        try:
            return int(field)
        except ValueError:
            error(line_number, field_number, Msg.mistyped_field, 'integer')
    elif field_type == 'q':
        if (field[0] == '"') != (field[-1] == '"'):  # pragma: nocover
            # This should never happen since this function is only called on things that _split() processed,
            # and _split() raises ParseError on malformed strings. The check is just here because parse_field is
            # part of the public API (for GmiValidator's use), and might tempt other users to misuse it.
            raise AssertionError("Malformed string")
        return field.strip('"')
    elif field_type == 't':
        if field[0] != '"' or field[-1] != '"':
            error(line_number, field_number, Msg.mistyped_field, 'quoted string')
        else:
            return field.strip('"')
    else:
        # Either the field is bad, or unimplemented. Both cases are programming errors.
        # This shouldn't happen in normal use.
        raise AssertionError(f"Unknown type '{field_type}'")  # pragma: nocover


def _split(line_number: int, line: str):
    """
    Split a line into its component pieces.

    Essentially just splits on spaces, but keeps quoted strings together. Inserts the line number at the beginning.

    Args:
        line_number: The line number.
        line: The line to split.

    Returns:
        A list of pieces in the line.
    """
    pieces = []
    start = 0
    line = line.strip()

    while start < len(line):
        if line[start] == ' ':
            start += 1
        elif line[start] == '"':
            end = line.find('"', start + 1)
            if end > start:
                pieces.append(line[start:end + 1])
                start = end + 1
            else:
                error(line_number, Msg.unclosed_string)
        else:
            end = line.find(' ', start + 1)
            if end > start:
                pieces.append(line[start:end])
                start = end + 1
            else:
                pieces.append(line[start:])
                start = len(line)

    if len(pieces) > MIN_NONEMPTY_LENGTH:
        pieces[CARD_INDEX] = pieces[CARD_INDEX].lower()

    return pieces


def _card_index_to_field_number(index: int):
    """
    Convert an index in a line into the field to be displayed to the user.

    Args:
        index: Index to convert.

    Returns:
        The converted index.
    """
    return index
