"""CbcScalarDatasetCreator class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import TreeNode

# 4. Local modules
from xms.mf6.components.cbc_dataset_creator_base import CbcDatasetCreatorBase
from xms.mf6.file_io.cbc_reader import CbcData, CbcReader


class CbcScalarDatasetCreator(CbcDatasetCreatorBase):
    """Creates datasets from the cbc file."""
    def __init__(self, cbc_filepath: Path | str, model_node: TreeNode, query):
        """Initializes the class.

        Args:
            cbc_filepath (str): File path of cbc file.
            model_node (TreeNode): Model tree node.
            query (xmsapi.dmi.Query): Object for communicating with GMS
        """
        super().__init__(cbc_filepath, model_node, query)

        self.dataset_writers = {}  # Dict of name: DatasetWriter

    def create(self) -> list[tuple[str, str]]:
        """Creates scalar datasets from data in the cbc file.

        I've tried to limit self._query to this function or self._setup().
        """
        try:  # Exceptions are raised and caught here if there are problems
            self._setup()
            self._create_datasets()
            self._add_datasets()
        except RuntimeError as e:
            return [('ERROR', str(e))]
        return []

    def _add_datasets(self):
        """Uses Query to tell GMS to add the datasets to the tree."""
        for _key, value in self.dataset_writers.items():
            self._query.add_dataset(value, folder_path=self._tree_folder)

    def _setup(self) -> None:
        """Return things we need to create the datasets.

        Returns:
            (str): Folder where datasets should be created.
        """
        self._base_setup()

    def _create_datasets(self):
        """Creates scalar datasets from data in the cbc file."""
        with CbcReader(self._cbc_filepath) as reader:
            for cbc_time_step in reader:
                for cbc_data in cbc_time_step.cbc_data:
                    if self._can_make_dataset_from_cbc_data(cbc_data):
                        self._append_to_datasets(cbc_data)
        self._appending_finished()

    def _dataset_name(self, cbc_data: CbcData):
        """Returns a good dataset name.

        header.text = "a character string ... indicating the flow type"

        txt2id2 = if IMETH=6, "a character string ... containing the package or model name for information in ID2"

        Returns:
            (str): See description.
        """
        if cbc_data.header.text and cbc_data.txt2id2:
            return f'{cbc_data.txt2id2}-{cbc_data.header.text}'
        elif cbc_data.header.text:
            return f'{cbc_data.header.text}'
        else:
            return f'{cbc_data.txt2id2}'

    def _can_make_dataset_from_cbc_data(self, cbc_data: CbcData) -> bool:
        """Returns True if the cbc data can be turned into a GMS scalar dataset.

        Args:
            cbc_data (CbcData): The data record.
        """
        # Skip FLOW_JA_FACE and anything that is sized bigger than the number of grid cells
        num_records = cbc_data.records_count()
        return num_records <= self._cell_count

    def _append_to_datasets(self, cbc_data: CbcData):
        """Appends the data to the H5 datasets.

        Args:
            cbc_data (CbcData): The data record.
        """
        dataset_name = self._dataset_name(cbc_data)
        dataset_writer = self.dataset_writers.get(
            dataset_name, self._init_dataset_writer(dataset_name, num_components=1)
        )
        self.dataset_writers[dataset_name] = dataset_writer

        values = cbc_data.dataset_values(self._cell_count)
        activity = [int(abs(v) != CbcReader.NO_DATA) for v in values]
        dataset_writer.append_timestep(time=cbc_data.header.totim, data=values, activity=activity)

    def _appending_finished(self):
        """Adds mins, maxs and times and closes the H5 file."""
        for writer in self.dataset_writers.values():
            writer.appending_finished()
