"""CsvEditorComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import orjson
from PySide2.QtCore import QDir
from PySide2.QtWidgets import QFileDialog

# 3. Aquaveo modules
# from xms.api.dmi import XmsEnvironment
from xms.core.filesystem import filesystem as fs

# 4. Local modules
from xms.mf6.components import dmi_util
from xms.mf6.components.package_component_base import PackageComponentBase
from xms.mf6.gui import gui_util


class CsvEditorComponent(PackageComponentBase):
    """A Dynamic Model Interface (DMI) component for a MODFLOW 6 solution text file viewer component."""
    def __init__(self, main_file):
        """Initializes the class.

        Args:
            main_file (str): The main file associated with this component.
        """
        super().__init__(main_file)
        # self.ftype = ''
        with open(main_file.strip('\"'), 'rb') as file:
            cards = orjson.loads(file.read())
        self.sim_uuid = cards['SIM_UUID']
        self.data_file = dmi_util.find_solution_file(
            cards.get('SOLUTION_FILE', ''), cards.get('SOLUTION_FILE_FULL', ''),
            cards.get('SOLUTION_FILE_RELATIVE', ''), main_file
        )
        # project_file = XmsEnvironment.xms_environ_project_path()
        # if project_file:
        #     self.data_file = fs.resolve_relative_path(os.path.dirname(project_file), self.data_file)

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        This component is overloading save_to_location so that we don't copy everything in solution directory
        to the new location on save.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE - happens when the tree item owner is duplicated. The new component will always be unlocked
                to start with.

                PACKAGE - happens when the project is being saved as a package. As such, all data must be copied and
                all data must use relative file paths.

                SAVE - happens when re-saving this project.

                SAVE_AS - happens when saving a project in a new location. This happens the first time we save a
                project.

                UNLOCK - happens when the component is about to be changed and it does not have a matching uuid folder
                in the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        return dmi_util.save_to_location_common(self.main_file, new_path, save_type)

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        # Nothing to do, override base class functionality
        return [], []

    def delete_event(self, lock_state):
        """This will be called when the component is deleted.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change or delete the files if
                locked.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        # Nothing to do, override base class functionality
        return [], []

    def open_dialog(self, query, params, win_cont):
        """Opens the package dialog.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        gui_util.open_file_in_default_app(self.data_file)
        return [], []

    def export_dialog(self, query, params, win_cont):
        """Opens the Save File dialog and exports the package to the indicated file.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        # Set the working directory of the file selector to be the location of the original text file if it is
        # not in the temp directory.
        if fs.is_somewhere_below_system_temp(self.main_file):
            start_dir = QDir.homePath()
        else:
            start_dir = os.path.dirname(self.main_file)

        filter_ext = 'CSV files (*.csv)'  # filter for export dialog
        filename, _ = QFileDialog.getSaveFileName(win_cont, "Save", dir=start_dir, filter=filter_ext)
        if filename:
            # Copy the file to the export location.
            fs.copyfile(self.main_file, filename)
        return [], []

    def update_displayed_cell_indices(self, model_mainfile, data):
        """Updates the cell ids used to display symbols.

        Args:
            model_mainfile: Path to the model main file.
            data: Package data class.

        Returns:
            An ActionRequest.
        """
        return None
