"""Runs the Flow Budget dialog."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import Optional

# 2. Third party modules
from PySide2.QtWidgets import QDialog, QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util, TreeNode
from xms.core.filesystem import filesystem as fs
from xms.guipy import file_io_util
from xms.guipy.dialogs import message_box

# 4. Local modules
from xms.mf6.components import dmi_util, sim_component
from xms.mf6.data import data_util
from xms.mf6.data.flow_budget_calculator import FlowBudgetCalculator
from xms.mf6.file_io.gwt.fmi_reader import FmiReader
from xms.mf6.gui.flow_budget_dialog import FlowBudgetDialog
from xms.mf6.misc.settings import Settings


def run_flow_budget(query: Query, win_cont: Optional[QWidget], sim_node: TreeNode, initial_model_node: TreeNode):
    """Runs the flow budget dialog when called by right-clicking on selected cells.

    Args:
        query: Object for communicating with GMS
        win_cont: The window container.
        sim_node: Simulation tree node.
        initial_model_node (TreeNode): The initial model tree node. None if right-clicking on selected cells.

    Returns:
        (tuple[bool, int]): True on OK, False if there was an error, and the number of decimals.
    """
    initial_model_node_passed = initial_model_node

    # Create model_to_cbc dict
    model_to_cbc = get_model_to_cbc_dict(sim_node, initial_model_node)
    if not any(model_to_cbc.values()):
        msg = 'Could not find any CBC solution. Has the solution been computed and read?'
        message_box.message_with_ok(parent=win_cont, message=msg)
        return False, None

    try:
        # Import here to avoid circular imports
        from xms.mf6.components.cbc_component import CbcComponent, DEFAULT_PREC
        precision: int = DEFAULT_PREC

        # Get flow budgets for each model
        flow_budgets: dict[TreeNode, dict | None] = {}
        for model_node, cbc_node in model_to_cbc.items():
            flow_budgets[model_node] = None
            if cbc_node:
                ugrid_uuid = dmi_util.ugrid_uuid_from_model_node(model_node)
                selected_cells = dmi_util.get_selected_cells(ugrid_uuid, query)
                grb_filepath = get_binary_grid_file(sim_node.name, model_node)
                dset_ptr_node = tree_util.descendants_of_type(
                    cbc_node.parent, xms_types=['TI_DATASET_PTR'], allow_pointers=True, only_first=True
                )
                if dset_ptr_node:
                    active_timestep_time = dmi_util.get_dataset_active_timestep_time(dset_ptr_node.uuid, query)
                else:
                    active_timestep_time = 0.0
                comp = CbcComponent(cbc_node.main_file)
                calculator = FlowBudgetCalculator(
                    cbc_filepath=comp.data_file,
                    selected_cells=selected_cells,
                    time=active_timestep_time,
                    grb_filepath=grb_filepath
                )
                flow_budget = calculator.calculate()
                flow_budgets[model_node] = flow_budget

                # Read precision (last one read gets used in the dialog)
                precision = Settings.get(comp.main_file, 'flow-budget-precision', DEFAULT_PREC)

                # If multiple models and this one has the active dataset, set initial_model_node
                if dset_ptr_node:
                    dset_node = tree_util.find_tree_node_by_uuid(query.project_tree, dset_ptr_node.uuid)
                    if len(model_to_cbc) > 1 and dset_node.is_active_item:
                        initial_model_node = model_node

        dialog = FlowBudgetDialog(
            flow_budgets=flow_budgets, initial_model_node=initial_model_node, prec=precision, parent=win_cont
        )
        if dialog.exec() == QDialog.Accepted:
            if initial_model_node_passed:  # Save precision to settings if coming from model or CBC component
                main_file = model_to_cbc[initial_model_node_passed].main_file
                Settings.set(main_file, 'flow-budget-precision', dialog.ui.spn_precision.value())
            return True, dialog.ui.spn_precision.value()
    except RuntimeError as e:
        message_box.message_with_ok(parent=win_cont, message=str(e))
    return False, None


def get_binary_grid_file(sim_name: str, model_node: TreeNode) -> Path:
    """Returns the binary grid file (.grb) path.

    MODFLOW seems to create the .grb file next to the .dis file, but only for the GWF (flow) models, not GWT
    (transport). So to find the .grb, we look in the directory where the mfsim.nam file is. If we don't find it, and
    the model is not a GWF, it's likely that the model is GWT and we are doing an uncouple simulation (if coupled, we
    should find the .grb). So then we look for the GWF simulation by using the FMI package, which will have the path to
    a file in the linked GWF model (e.g. budget or head file).

    Args:
        sim_name: Name of the simulation, so we know what folder to search.
        model_node: Model tree node, so we can get the FMI package, if necessary.
    """
    mfsim_nam = Path(sim_component.saved_sim(sim_name))
    files = list(mfsim_nam.parent.glob('*.grb'))
    if files:
        return files[0]

    def ensure_true(condition):
        """Helper function to avoid many nested if statements, and to raise an exception with 1 line instead of 2."""
        if not condition:
            raise RuntimeError(
                'No binary grid file found (.grb). This file is required for the Flow Budget. If the '
                ' "NOGRB" option is on in the GWF DIS* package, turn it off and rerun MODFLOW.'
            )

    # If this is not a GWF model, try to look in the GWF model
    ensure_true(model_node.unique_name != 'GWF6')

    # Find FMI package
    fmi_node = tree_util.descendants_of_type(model_node, unique_name='FMI6', only_first=True)
    ensure_true(fmi_node)

    # Read FMI package
    reader = FmiReader()
    data = reader.read(fmi_node.main_file)
    ensure_true(data)

    # Get a path to a file in the GWF model
    block = 'PACKAGEDATA'
    filename = data.list_blocks.get(block)
    ensure_true(filename)
    column_names, column_types, _ = data.get_column_info(block)
    df = data.read_csv_file_into_dataframe(block, filename, column_names, column_types)
    ensure_true(df is not None and len(df) > 0)
    path = df.iloc[0]['FNAME']  # Will be path to a GWF file (e.g. GWFBUDGET, or GWFHEAD)
    ensure_true(path)

    # Should be a relative path. Resolve it to a full path
    full_path = fs.resolve_relative_path(str(mfsim_nam), path)
    ensure_true(full_path)
    path = Path(full_path).parent
    ensure_true(path)

    # Search for a .grb file in the path, or in the parent path
    files = list(path.glob('*.grb'))
    if files:
        return files[0]
    else:
        # Try looking up one directory
        files = list(path.parent.glob('*.grb'))
        if files:
            return files[0]

    # We couldn't find it. Raise the exception
    ensure_true(False)


def get_model_to_cbc_dict(sim_node: TreeNode, model_node: TreeNode | None) -> dict[TreeNode, TreeNode]:
    """Returns a dict of model node -> CBC component tree node.

    Args:
        sim_node: Simulation tree node.
        model_node: The only model to consider. If None, consider all models.

    Returns:
        See description.
    """
    # Get all the CBC component nodes
    types = ['TI_COMPONENT']
    cbc_nodes = tree_util.descendants_of_type(sim_node, xms_types=types, unique_name='CBC', model_name='MODFLOW 6')
    if not cbc_nodes:
        cbc_nodes = tree_util.descendants_of_type(sim_node, xms_types=types, unique_name='CBB', model_name='MODFLOW 6')

    # Create dict with model nodes and initialize CBC nodes to None initially. Also create helper dics.
    ftypes = data_util.model_ftypes()
    model_to_cbc: dict[TreeNode, TreeNode | None] = {}
    model_name_to_uuid: dict[str, str] = {}
    model_uuid_to_node: dict[str, TreeNode] = {}
    for c in sim_node.children:
        if c.unique_name in ftypes and (model_node is None or c == model_node):
            model_to_cbc[c] = None
            model_name_to_uuid[c.name] = c.uuid
            model_uuid_to_node[c.uuid] = c

    # Match the CBC nodes to the model nodes
    for cbc_node in cbc_nodes:
        json_data = file_io_util.read_json_file(cbc_node.main_file)
        model_uuid = json_data.get('MODEL_UUID', '')
        if model_uuid and model_uuid in model_uuid_to_node:
            model_to_cbc[model_uuid_to_node[model_uuid]] = cbc_node
        else:
            if 'SOLUTION_FILE_FULL' in json_data and json_data['SOLUTION_FILE_FULL']:
                stem = Path(json_data['SOLUTION_FILE_FULL']).stem
                if stem in model_name_to_uuid:
                    model_uuid = model_name_to_uuid[stem]
                    if model_uuid in model_uuid_to_node:
                        model_to_cbc[model_uuid_to_node[model_uuid]] = cbc_node
    return model_to_cbc
