"""ModelComponentBase class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
from typing import List, Optional

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QApplication, QDialog, QWidget
from typing_extensions import override

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query
from xms.api.tree import tree_util
from xms.components.display.display_options_io import (
    read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.constraint import Grid
from xms.core.filesystem import filesystem as fs
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.dialogs import xms_parent_dlg
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.widgets.category_display_options_list import Columns

# 4. Local modules
from xms.mf6.components import (
    default_package_creator, dis_grid_comparer, dmi_util, flow_budget_runner, new_sim_dialog_runner
)
from xms.mf6.components.package_component_base import PackageComponentBase
from xms.mf6.data.base_file_data import BaseFileData
from xms.mf6.data.mfsim_data import MfsimData
from xms.mf6.data.model_data_base import ModelDataBase
from xms.mf6.file_io import io_factory, io_util
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.gui import gui_util
from xms.mf6.gui.cell_properties_dialog import CellPropertiesDialog
from xms.mf6.gui.dialog_input import DialogInput
from xms.mf6.misc.settings import Settings


class ModelComponentBase(PackageComponentBase):
    """Base class for Gwf/Gwt components."""

    CELL_PROPERTIES_ACTIVE = 'cell-properties-active'  # To remember which package is active in cell properties dialog

    def __init__(self, main_file):
        """Initializes the class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.ftype = ''  # package ftype
        self.dialog = None  # dialog class
        self.link_param = ''
        self.model_str = ''
        self._display_option_descriptions = {}  # new -> original descriptions

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (tuple): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)
        return new_main_file, messages, action_requests

    def unlink_event(self, unlinks, lock_state):
        """This will be called when a coverage, or a ugrid, or another component is unlinked from this component.

        Args:
            unlinks (list of str): A list of UUIDs as strings representing the objects being unlinked.
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        # Send back an action request so that the lists are redrawn
        action = ActionRequest(
            main_file=self.main_file,
            modality='no_dialog',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='unlink_ugrid'
        )
        return [], [action]

    def link_event(self, link_dict, lock_state):
        """This will be called when one or more coverages, ugrids, or other components are linked to this component.

        Args:
            link_dict (dict): A dictionary with keys being UUIDs as strings representing the objects being linked into
                this component. The values of this dictionary are a list of strings of the parameter names of the
                "takes" from the XML that this is a part of.
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        actions = []
        for link_uuid, link_xml_params in link_dict.items():
            for xml_param in link_xml_params:
                if xml_param == self.link_param:  # We are linking a new UGrid to the GWF/GWT model
                    action = ActionRequest(
                        main_file=self.main_file,
                        modality='no_dialog',
                        class_name=self.class_name,
                        module_name=self.module_name,
                        method_name='update_on_ugrid_link',
                        parameters={'ugrid_uuid': link_uuid}
                    )
                    actions.append(action)
                    break

        return [], actions

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        with open(self.main_file, 'w') as _:
            pass

        new_sim_action = ActionRequest(
            main_file=self.main_file,
            modality='modal',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='open_new_sim_dialog'
        )
        messages = []
        action_requests = [new_sim_action]
        return messages, action_requests

    def _make_duplicate_display_action(self, new_main_file: str) -> ActionRequest:
        """Create an action request to update the display of all our displayable children after duplicate.

        Arguments:
            new_main_file: Path to the duplicated component's main file

        Returns:
            (ActionRequest): See description.
        """
        return ActionRequest(
            main_file=new_main_file,
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='_duplicate_display_action',
            comp_uuid=io_util.uuid_from_path(new_main_file)
        )

    def open_new_sim_dialog(self, query, params, win_cont):
        """Opens the New Simulation dialog and creates the model files indicated.

        Args:
            query: Object for communicating with GMS
            params: Generic map of parameters. Unused in this case.
            win_cont: The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        help_id = gui_util.help_id_from_key('NewSimDialog')
        return new_sim_dialog_runner.run_dialog(
            self.main_file, query, params, win_cont, model_str=self.model_str, help_id=help_id
        )

    @override
    def get_project_explorer_menus(self, main_file_list):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (list of str): A list of the main files of the selected components of this type.

        Returns:
            menu_items (list of xmsapi.dmi.MenuItem): A list of menus and menu items to be shown. Note
            that this list can have objects of type xmsapi.dmi.Menu as well as xmsapi.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        if len(main_file_list) > 1 or not main_file_list:
            return []  # Multi-select or nothing selected

        menu_list = [None]
        self._add_tree_menu_command('Open...', 'open_dialog', '', menu_list)
        self._add_tree_menu_command('Display Options...', 'display_options', '', menu_list)
        self._add_tree_menu_command('Flow Budget...', '_flow_budget_from_tree', '', menu_list)
        menu_list.append(None)
        return menu_list

    def _add_selected_cells_menu_command(self, menu_text, method_name, selection, menu_list):
        dmi_util.add_selected_cells_menu_command(
            menu_text=menu_text,
            method_name=method_name,
            main_file=self.main_file,
            class_name=self.class_name,
            module_name=self.module_name,
            menu_list=menu_list,
            selection=selection
        )

    def get_double_click_actions(self, lock_state):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        messages = []
        actions = []

        open_action = ActionRequest(
            main_file=self.main_file,
            modality='modal',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='open_dialog',
            parameters={'main_file': self.main_file}
        )
        actions.append(open_action)

        return messages, actions

    def update_on_ugrid_link(self, query: Query, params):
        """Updates all cellid files for all packages because UGrid has been linked.

        Args:
            query: Query with a context at the component instance level.
            params: ActionRequest parameters

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        messages = []
        actions = []
        try:
            # Get project explorer tree node
            model_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())

            # Make sure grid is good
            ugrid_uuid = dmi_util.ugrid_uuid_from_model_node(model_node)
            dogrid = query.item_with_uuid(ugrid_uuid)  # Get the data_objects UGrid
            cogrid, error_messages = default_package_creator.read_cogrid(dogrid.cogrid_file)
            if not cogrid or error_messages:
                messages.append(error_messages)
                return messages, actions

            # Read mfsim
            mfsim, model, package = self.read_sim(query)  # can we just do this?

            # Set grids
            mfsim.xms_data.set_dogrid(model.tree_node.uuid, dogrid)
            mfsim.xms_data.set_cogrid(model.tree_node.uuid, cogrid)

            # Compare DIS* package to grid
            _get_dis_ugrid_mismatches(model, cogrid, messages)

            # Update display
            # Update displayed cell indices
            for package in model.packages:
                package.update_displayed_cell_indices()
            self._update_display(cogrid.uuid, model_node, '')

        except Exception as e:
            raise e

        return messages, actions

    def unlink_ugrid(self, query, params):
        """The UGrid was unlinked so update the display.

        This is cannot be a static method. It must be an instance method because it is an ActionRequest method.

        Args:
            query (xmsapi.dmi.Query): Query with a context at the component instance level.
            params: Unused.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        model_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())
        ugrid_uuid = dmi_util.ugrid_uuid_from_model_node(model_node)

        main_file = Path(self.main_file)
        if not main_file.parent.is_dir():
            return [], []

        filename = main_file.parent / 'cell.display_indices'
        write_display_option_ids(str(filename), [])
        self._update_display(ugrid_uuid=ugrid_uuid, model_node=model_node, override_file=filename)

        return [], []

    def display_options(self, query, params: Optional[List[dict]], win_cont: Optional[QWidget]):
        """Shows the display options dialog.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        del params  # Unused parameter
        dlg, categories_list_files, ugrid_uuid = self._setup_display_options_dialog(query, win_cont)
        xms_parent_dlg.add_process_id_to_window_title(dlg)
        if dlg.exec() == QDialog.Accepted:
            # write files
            category_lists = dlg.get_category_lists()
            for category_list in category_lists:
                filename = categories_list_files[category_list.uuid]
                self._restore_original_descriptions(category_list)
                write_display_options_to_json(filename, category_list)
                if ugrid_uuid:
                    self.display_option_list.append(XmsDisplayMessage(file=filename, edit_uuid=ugrid_uuid))

        messages: list[tuple[str]] = []
        actions: list[ActionRequest] = []
        return messages, actions

    def _restore_original_descriptions(self, category_list):
        """Restore the original descriptions before writing to disk.

        Args:
            category_list (CategoryDisplayOptionList): A CategoryDisplayOptionList.
        """
        for category in category_list.categories:
            category.description = self._display_option_descriptions[category.description]

    def _setup_category_lists(self, query):
        """Walks the tree to create and return things needed by the dialog.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS

        Returns:
            (tuple[list[CategoryDisplayOptionList], dict[str, str], str]): See description.
        """
        model_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())
        categories_list = []
        categories_list_files = {}
        displayable_ftypes = BaseFileData.displayable_ftypes()
        ugrid_uuid = ''
        for child in model_node.children:
            ftype = child.unique_name
            if child.item_typename == 'TI_COMPONENT' and ftype in displayable_ftypes:
                if ftype == 'RCH6' and io_util.file_has_readasarrays(child.main_file):
                    continue
                disp_opts_files = dmi_util.get_display_option_filenames(child.main_file)
                for disp_opts_file in disp_opts_files:
                    json_dict = read_display_options_from_json(disp_opts_file)

                    # Update the description with the name from the project tree
                    description = json_dict['categories'][0]['description']
                    # Format is for example: 'tree_name (River - RIV6)'
                    new_description = f'{child.name} ({description} - {child.unique_name})'
                    json_dict['categories'][0]['description'] = new_description
                    self._display_option_descriptions[new_description] = description

                    categories = CategoryDisplayOptionList()
                    categories.from_dict(json_dict)
                    categories_list.append(categories)
                    categories_list_files[categories.uuid] = disp_opts_file
            elif child.item_typename == 'TI_UGRID_PTR':  # Snag the UGrid UUID
                ugrid_uuid = child.uuid

        return categories_list, categories_list_files, ugrid_uuid

    def _setup_display_options_dialog(self, query, win_cont):
        """Sets up the display options dialog.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            win_cont (QWidget): The window container.

        Returns:
            (tuple[CategoryDisplayOptionsDialog, dict[str, str], str]): The dialog and other needed data
        """
        categories_list, categories_list_files, ugrid_uuid = self._setup_category_lists(query)
        dlg = CategoryDisplayOptionsDialog(categories_list, win_cont, show_on_off=False)
        dlg.ui.display_options.ui.table_view_categories.setColumnHidden(Columns.FONT_CHECK, True)
        dlg.ui.display_options.ui.table_view_categories.setColumnHidden(Columns.FONT, True)
        dlg.setModal(True)
        dlg.help_getter = gui_util.help_getter(gui_util.help_id_from_key('DisplayOptions'))
        locked = query.current_item().locked
        dlg.set_read_only(locked)
        return dlg, categories_list_files, ugrid_uuid

    def open_dialog(self, query, params, win_cont):
        """Opens the GWF model dialog.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        del params  # Unused parameter
        messages = []
        actions = self._open_dialog(query, win_cont)
        return messages, actions

    def _open_dialog(self, query: Query, win_cont: QWidget | None) -> list[ActionRequest]:
        """Opens the model dialog.

        Args:
            query: Object for communicating with GMS
            win_cont: The window container.

        Returns:
            list[ActionRequest]: List of action requests.
        """
        actions = []

        mfsim, model, _package = self.read_sim(query)
        mfsim_dir = os.path.dirname(mfsim.filename)

        # Do the dialog
        locked = query.current_item().locked
        help_id = gui_util.help_id_from_key(self.ftype)
        dlg_input = DialogInput(data=model, locked=locked, help_id=help_id)
        dialog = self.dialog(dlg_input, win_cont)
        dialog.setModal(True)
        if dialog.exec() == QDialog.Accepted and not locked:
            # Write the package
            writer_options = WriterOptions(
                mfsim_dir=mfsim_dir,
                use_open_close=True,
                use_input_dir=False,
                use_output_dir=False,
                dmi_sim_dir=os.path.join(mfsim_dir, '../..')
            )
            writer = io_factory.writer_from_ftype(self.ftype, writer_options)
            QApplication.setOverrideCursor(Qt.WaitCursor)
            writer.write(dialog.dlg_input.data)
            QApplication.restoreOverrideCursor()

        return actions

    def get_initial_display_options(self, query, params):
        """Gets called to load default display options if 'use_display' attribute is True in the component definition.

        Args:
            query (xmsapi.dmi.Query): Query with a context at the component instance level.
            params: Unused.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        # Get project explorer tree
        model_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())
        ugrid_uuid = dmi_util.ugrid_uuid_from_model_node(model_node)

        if not ugrid_uuid:
            return [], []

        if params and 'filenames' in params[0]:
            display_files = params[0]['filenames']
            self._update_display_list_files(ugrid_uuid, display_files)
        else:
            self._update_display(ugrid_uuid, model_node, '')

        return [], []

    def get_display_menus(self, selection, lock_state, id_files):
        """This will be called when right-click menus in the main display area of XMS are being created.

        Args:
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                The value of the dictionary is a list of IntegerLiteral ids of the selected feature objects.
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.
            id_files (dict): Key is entity type string, value is tuple of two str where first is the file
                location of the XMS coverage id binary file. Second is file location of the component coverage id binary
                file. Only applicable for coverage selections. File will be deleted after event. Copy if need to
                persist.

        Returns:
            menu_items (list of xmsapi.dmi.MenuItem): A list of menus and menu items to be shown. Note
            that this list can have objects of type xmsapi.dmi.Menu as well as xmsapi.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        menu_list = [None]  # None == spacer

        if 'CELL' in selection:
            self._add_selected_cells_menu_command('Cell Properties...', 'cell_properties', selection, menu_list)
            self._add_selected_cells_menu_command('Flow Budget...', '_flow_budget_from_cells', selection, menu_list)
        return menu_list

    def _update_display(self, ugrid_uuid, model_node, override_file: str | Path):
        """Update the display lists of all child components.

        Args:
            ugrid_uuid: UUID of the UGrid.
            model_node (TreeNode): The GWF project explorer tree node
            override_file: file used to override disp_opts_file. Used when unlinking a ugrid so
             that no symbols are displayed.
        """
        if override_file:
            override_file = str(override_file)
        displayable_types = BaseFileData.displayable_ftypes()
        for child in model_node.children:
            ftype = child.unique_name
            if ftype in displayable_types:
                disp_opts_files = dmi_util.get_display_option_filenames(child.main_file)
                for disp_opts_file in disp_opts_files:
                    if override_file:
                        dest = os.path.join(os.path.dirname(disp_opts_file), 'cell.display_indices')
                        fs.copyfile(override_file, dest)
                    self.display_option_list.append(XmsDisplayMessage(file=disp_opts_file, edit_uuid=ugrid_uuid))

    def _cell_properties_packages_exist(self, mfsim: MfsimData):
        """Returns True if any packages supported by the CellProperties dialog exist.

        Args:
            mfsim: The simulation data.

        Returns:
             See description.
        """
        for model in mfsim.models:
            for child in model.tree_node.children:
                ftype = child.unique_name
                if ftype in CellPropertiesDialog.supported_ftypes:
                    return True
        return False

    def _active_cell_properties_package(self, sim: MfsimData):
        """Returns the package data we want to be active in the cell properties dialog."""
        active_uuid = Settings.get(self.main_file, self.CELL_PROPERTIES_ACTIVE)
        if active_uuid:
            for model in sim.models:
                for package in model.packages:
                    if (
                        package.ftype in CellPropertiesDialog.supported_ftypes and  # noqa: W504
                        package.tree_node.uuid == active_uuid
                    ):
                        return package
        else:
            return None

    def _save_active_cell_properties(self, dialog: CellPropertiesDialog) -> None:
        """Saves which package in cell properties is active, so we can remember it next time."""
        uuid = dialog.current_package_uuid()
        Settings.set(self.main_file, self.CELL_PROPERTIES_ACTIVE, uuid)

    def cell_properties(self, query, params: Optional[List[dict]], win_cont: Optional[QWidget]):
        """Runs the cell properties dialog.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params ([dict]): ActionRequest parameters
            win_cont (PySide2.QtWidgets.QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        # Get project explorer tree
        model_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())

        # Get the selected cell indices
        ugrid_uuid = dmi_util.ugrid_uuid_from_model_node(model_node)
        cell_idxs = dmi_util.get_selected_cells(ugrid_uuid, query)
        if len(cell_idxs) == 0:
            return [('INFO', 'No cells selected. Select one or more cells.')], []

        # Get files and directories
        sim_node = model_node.parent
        mfsim_nam = query.item_with_uuid(
            item_uuid=sim_node.uuid, model_name='MODFLOW 6', unique_name='Sim_Manager'
        ).main_file
        mfsim_dir = os.path.dirname(mfsim_nam)
        dmi_sim_dir = os.path.normpath(os.path.join(mfsim_dir, '../..'))
        reader = io_factory.reader_from_ftype('MFSIM6')
        mfsim = reader.read(mfsim_nam, sim_node, query)

        # Abort if no packages have cell properties
        if not self._cell_properties_packages_exist(mfsim):
            return [('INFO', 'No defined packages that have cell properties.')], []

        # Create the cell properties dialog
        active_package = self._active_cell_properties_package(mfsim)
        help_id = gui_util.help_id_from_key('CellPropertiesDialog')
        dlg_input = _create_cell_properties_dlg_input(active_package, cell_idxs, ugrid_uuid, query, help_id, mfsim)
        dialog = _create_cell_properties_dlg(dlg_input, win_cont)

        # Run the cell properties dialog
        if dialog.exec() != QDialog.Accepted:
            return [], []

        # Save data and update display lists for packages in dialog
        self._save_active_cell_properties(dialog)
        for model in mfsim.models:
            for data in model.packages:
                if data.ftype not in CellPropertiesDialog.supported_ftypes:
                    continue
                if dmi_util.is_locked(data.tree_node, query):
                    continue

                writer_options = WriterOptions(mfsim_dir=mfsim_dir, use_open_close=True, dmi_sim_dir=dmi_sim_dir)
                writer = io_factory.writer_from_ftype(data.ftype, writer_options)
                if writer:
                    writer.write(data)
                    data.update_displayed_cell_indices()
                    display_files = dmi_util.get_display_option_filenames(data.filename)
                    self._update_display_list_files(ugrid_uuid, display_files)
        return [], []

    def _flow_budget(self, query: Query, params: Optional[List[dict]], win_cont: Optional[QWidget], from_cells: bool):
        """Do Flow Budget menu item.

        Args:
            query: Object for communicating with GMS
            params: ActionRequest parameters
            win_cont: The window container.
            from_cells: True if called from right-clicking on selected cells.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        model_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())
        sim_node = model_node.parent
        flow_budget_runner.run_flow_budget(query, win_cont, sim_node, model_node if not from_cells else None)
        return [], []

    def _flow_budget_from_tree(self, query: Query, params: Optional[List[dict]], win_cont: Optional[QWidget]):
        """Called when Flow Budget menu is called from the Project Explorer.

        Args:
            query: Object for communicating with GMS
            params: ActionRequest parameters
            win_cont: The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        return self._flow_budget(query, params, win_cont, from_cells=False)

    def _flow_budget_from_cells(self, query: Query, params: Optional[List[dict]], win_cont: Optional[QWidget]):
        """Called when Flow Budget menu is called from right-clicking on selected cells.

        Args:
            query: Object for communicating with GMS
            params: ActionRequest parameters
            win_cont: The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        return self._flow_budget(query, params, win_cont, from_cells=True)

    def _update_display_list_files(self, ugrid_uuid, display_files):
        """Update a specific set of child component display lists.

        Args:
            ugrid_uuid (str): UUID of the linked UGrid
            display_files (list): List of the component display option files to update.
        """
        for filename in display_files:
            self.display_option_list.append(XmsDisplayMessage(file=filename, edit_uuid=ugrid_uuid))


def _get_dis_ugrid_mismatches(model: ModelDataBase, cogrid: Grid, messages: list[tuple[str, str]]) -> None:
    """Compares the DIS* package to the grid and fills in messages with any mismatches.

    Args:
        model: The model.
        cogrid: The constrained grid.
        messages: list of error messages
    """
    errors = dis_grid_comparer.get_model_mismatches(model=model, cogrid=cogrid)
    for error in errors:
        messages.append(('WARNING', error))


def _create_cell_properties_dlg_input(active_package, cell_idxs, ugrid_uuid, query, help_id, mfsim) -> DialogInput:
    """To make testing easier, creates and returns the DialogInput.

    Args:
        active_package: The package to first show in the dialog.
        cell_idxs: Selected cell indices.
        ugrid_uuid: UUID of the UGrid.
        query: Object for communicating with GMS
        help_id: The second part of the wiki help line on the above page (after the '|').
        mfsim: MfsimData object.

    Returns:
        See description.
    """
    dlg_input = DialogInput(
        data=active_package,
        locked=False,
        selected_cells=cell_idxs,
        filter_on_selected_cells=True,
        ugrid_uuid=ugrid_uuid,
        query=query,
        help_id=help_id,
        mfsim=mfsim
    )
    return dlg_input


def _create_cell_properties_dlg(dlg_input: DialogInput, win_cont: QWidget | None) -> CellPropertiesDialog:
    """To make testing easier, creates and returns the CellPropertiesDialog.

    Args:
        dlg_input: Information needed by the dialog.
        win_cont: The window container.

    Returns:
        See description.
    """
    return CellPropertiesDialog(dlg_input, win_cont)
