"""Data utility functions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.testing import tools

# 4. Local modules
from xms.mf6.data.grid_info import DisEnum, GridInfo
from xms.mf6.misc import util

_uuids: dict[str, str] = {}  # For testing so we can reuse the same uuids


def uuid_key(package) -> str:
    """Return the key for the _uuids dict.

    Args:
        package: The package.
    """
    if package.ftype in {'IMS6', 'EMS6'}:
        key2 = f':{package.fname}'
    else:
        key2 = f':{package.model.ftype}' if package.model and package.ftype not in model_ftypes() else ''
    return f'{fix_ftype(package.ftype, with_the_a=package.readasarrays)}{key2}'


def save_uuid(package, uuid_str: str) -> None:
    """Saves the uuid for later testing purposes.

    Args:
        package: The package.
        uuid_str: The uuid string.
    """
    key = uuid_key(package)
    _uuids[key] = uuid_str


def get_saved_uuids() -> dict[str, str]:
    """Return the uuids that were saved for testing purposes."""
    return _uuids


def new_uuid(package) -> str:
    """Returns a new, random uuid, or, if testing, can be patched to return the uuid associated with the ftype.

    Args:
        package: The package data.

    Returns:
        See description.
    """
    uuid_str = tools.new_uuid()
    save_uuid(package, uuid_str)
    return uuid_str


def advanced_transport_time_series_columns(package) -> list[int]:
    """Returns a list of the column indices that can contain time series.

    For the LKT, MWT, SFT, and UZT packages since they're all the same.

    Returns:
        List of indices of columns that can contain time series.
    """
    names, _, _ = package.get_column_info('PACKAGEDATA')
    start = names.index('STRT')
    end = len(names) - 1 if package.options_block.has('BOUNDNAMES') else len(names)
    columns = list(range(start, end))
    return columns


def get_id_column_dict(grid_info):
    """Returns cell id column info (lay, row, col etc) based on dis package.

    Args:
        grid_info (GridInfo): Number of rows, cols etc.

    Returns:
        columns (dict): Dict of column name -> (type, default value)
    """
    if not grid_info:
        grid_info = GridInfo()

    if grid_info.dis_enum == DisEnum.DIS:
        columns = {
            'LAY': (np.int32, 1),
            'ROW': (np.int32, 1),
            'COL': (np.int32, 1),
        }
    elif grid_info.dis_enum == DisEnum.DISV:
        columns = {
            'LAY': (np.int32, 1),
            'CELL2D': (np.int32, 1),
        }
    elif grid_info.dis_enum == DisEnum.DISU:
        columns = {
            'CELLID': (np.int32, 1),
        }
    else:
        raise AssertionError()  # Error!

    return columns


def add_boundname_columns_to_dict(options_block, columns):
    """If the BOUNDNAMES option is on, adds the BOUNDNAME column to the dict.

    Args:
        options_block (OptionsBlock): The options block.
        columns (dict of str -> type): Column names -> column types.
    """
    if options_block.has('BOUNDNAMES'):
        columns['BOUNDNAME'] = (object, '')


def extension_from_ftype(ftype: str) -> str:
    """Returns the file extension (e.g. ".wel") associated with the ftype.

    If the ftype is not recognized, it just returns the ftype.

    Args:
        ftype (str): The file type used in the GWF name file (e.g. 'WEL6')

    Returns:
        (str): See description.
    """
    ftype_extension_dict = {  # Things that don't fit the general pattern
        'MFSIM6': '.nam',
        'GWF6': '.nam',
        'GWT6': '.nam',
        'GWE6': '.nam',
        'PRT6': '.nam',
        'ZONE6': '.zon',
        'EVTA6': '.evta',
        'RCHA6': '.rcha',
    }
    extension = ftype_extension_dict.get(ftype)
    if not extension:
        return f'.{ftype.replace("-", "").replace("6", "").lower()}'
    return ftype_extension_dict.get(ftype, ftype)


def filter_from_ftype(ftype: str) -> str:
    """Return the file filter (e.g. 'WEL6 Files (*.wel)') for use in the open/save dialogs.

    Args:
        ftype: The ftype.

    Returns:
        See description.
    """
    if ftype == 'MFSIM6':
        return 'Simulation Name Files (mfsim.nam)'
    elif ftype == 'ZONE6':
        return 'ZONE Files (*.zon)'
    else:
        return f'{ftype} Files (*{extension_from_ftype(ftype)})'


def sim_child_ftypes() -> set[str]:
    """Get the ftypes of components that are owned by the simulation but are not models.

    Returns:.
        (set): See description
    """
    ftypes = exchange_ftypes()
    return ftypes.union({'TDIS6', 'IMS6', 'EMS6'})


def model_class_module() -> dict[str, tuple]:
    """Get a mapping of model ftype to component class name and module name pair.

    Returns:
        (dict[str, tuple]): See description
    """
    return {
        'GWF6': ('GwfComponent', 'xms.mf6.components.gwf.gwf_component'),
        'GWT6': ('GwtComponent', 'xms.mf6.components.gwt.gwt_component'),
        'GWE6': ('GweComponent', 'xms.mf6.components.gwe.gwe_component'),
    }


def model_ftypes() -> set[str]:
    """Return the set of model ftypes (i.e. 'GWF6', 'GWT6' etc).

    Returns:
        See description.
    """
    if util.PRT_INTERFACE:
        return {'GWF6', 'GWT6', 'GWE6', 'PRT6'}
    else:
        return {'GWF6', 'GWT6', 'GWE6'}


def dis_ftypes() -> set[str]:
    """Return the set of dis* package ftypes (i.e. 'DIS6', 'DISV6', 'DISU6').

    Returns:
        See description.
    """
    return {'DIS6', 'DISV6', 'DISU6'}


def exchange_ftypes() -> set[str]:
    """Get the ftypes of exchanges (e.g. 'GWF6-GWF6', 'GWF6-GWT6').

    Returns:.
        (set): See description
    """
    return {'GWF6-GWF6', 'GWF6-GWT6', 'GWF6-GWE6', 'GWF6-PRT6', 'GWT6-GWT6', 'GWE6-GWE6'}


def readasarrays_ftypes(with_the_a: bool) -> set[str]:
    """Return the set of package ftypes associated with packages that can have READASARRAYS as an option.

    Args:
        with_the_a: True if you want them with the 'A', e.g. EVTA6 instead of EVT6.

    Returns:
        See description.
    """
    if with_the_a:
        return {'EVTA6', 'RCHA6'}  # SPC and SPT too, but we don't support them yet
    else:
        return {'EVT6', 'RCH6'}  # SPC and SPT too, but we don't support them yet


def fix_ftype(ftype: str, with_the_a: bool) -> str:
    """Return the set of package ftypes associated with packages that can have READASARRAYS as an option.

    Args:
        ftype: The ftype.
        with_the_a: True if you want it with the 'A', e.g. EVTA6 instead of EVT6.

    Returns:
        See description.
    """
    if with_the_a and ftype in readasarrays_ftypes(with_the_a=False):
        # Add the A
        return ftype[:-1] + 'A6'  # e.g. 'EVT6' to 'EVTA6'
    elif not with_the_a and ftype in readasarrays_ftypes(with_the_a=True):
        # Remove the A
        return ftype[:-2] + '6'
    return ftype


def compute_auto_names(packages, prefix: str) -> list[str]:
    """Returns list of auto-computed base file names, appending a number if needed to be unique.

    Filenames are of the form "prefix.wel" or "prefix_1.wel", "prefix_2.wel" if there are multiple.

    Args:
        packages: Collection of packages, or models.
        prefix (str): Filename prefix.

    Returns:
        (str): See description.
    """
    # Count how many we have of each ftype
    ftype_totals = {}
    for package in packages:
        # Call fix_ftype() to get the array-based version of the ftype for those packages (i.e. 'EVTA6', not 'EVT6')
        ftype = fix_ftype(package.ftype, with_the_a=package.readasarrays)
        ftype_totals[ftype] = ftype_totals.get(ftype, 0) + 1

    # Create filenames
    filenames = []
    ftype_counts = {}
    for package in packages:
        ftype = fix_ftype(package.ftype, with_the_a=package.readasarrays)
        if ftype in model_ftypes():
            # The model name can not have spaces, and wrapping it in single quotes does not work with MF6
            filenames.append(auto_file_name(package.mname.replace(' ', '_'), ftype))
        else:
            filenames.append(_compute_auto_name(ftype, ftype_counts, ftype_totals, prefix))
    return filenames


def _compute_auto_name(ftype, ftype_counts, ftype_totals, prefix) -> str:
    if ftype_totals[ftype] > 1:
        count = ftype_counts.get(ftype, 0)
        extension = extension_from_ftype(ftype)
        ftype_counts[ftype] = count + 1
        return f'{prefix}_{count + 1}{extension}'
    else:
        return auto_file_name(prefix, ftype)


def auto_file_name(prefix, ftype_or_extension):
    """Returns a filename of the form 'prefix.wel'.

    If ftype_or_extension starts with a '.', we assume it's an extension. Otherwise, we assume it's an ftype and we
    look up the extension.

    Args:
        ftype_or_extension (str): The ftype (e.g. 'WEL6'), or an extension starting with '.'
        prefix (str): Filename prefix.

    Returns:
        (str): See description.
    """
    if ftype_or_extension.startswith('.'):
        return f'{prefix}{ftype_or_extension}'
    else:
        return f'{prefix}{extension_from_ftype(ftype_or_extension)}'


def get_budget_file_name(model):
    """Returns the budget filename from the OC package."""
    name = ''
    oc_list = model.packages_from_ftype('OC6')
    if oc_list:
        name = oc_list[0].get_budget_file_name()
    return name


def get_dv_filename(model):
    """Returns the head filename from the OC package."""
    name = ''
    oc_list = model.packages_from_ftype('OC6')
    if oc_list:
        name = oc_list[0].get_dv_filename()
    return name
