"""BuyData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.base_file_data import BaseFileData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import gui_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxField


class BuyData(BaseFileData):
    """Data class to hold the info from the BUY package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'BUY6'
        self.list_blocks = {'PACKAGEDATA': ''}  # (list blocks -> filename)

    # @overrides
    def get_column_info(self, block, use_aux=True):
        """Returns column names, types, and defaults.

        The columns depend on the DIS package in use and the AUX variables.
        The package specific and AUX columns are type object because they
        might contain time series strings.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        return self.package_column_info()

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info(block)
        return {
            names.index('IRHOSPEC'): 'Species number',
            names.index('DRHODC'): 'Slope of the density-concentration line',
            names.index('CRHOREF'): 'Reference concentration',
            names.index('MODELNAME'): 'Name of GWT model used to simulate a species',
            names.index('AUXSPECIESNAME'):
                'Name of an auxiliary variable in a GWF stress package that will be'
                ' used for this species to calculate a density value.',
        }

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        You should override this method.

        Args:
            block (str): Name of the block.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        columns = {
            'IRHOSPEC': (np.int32, 1),
            'DRHODC': (np.float64, 0.0),
            'CRHOREF': (np.float64, 0.0),
            'MODELNAME': (object, ''),
            'AUXSPECIESNAME': (object, ''),
        }
        return gui_util.column_info_tuple_from_dict(columns)

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Buoyancy (BUY) Package'

    # def map_info(self, feature_type):
    #     """Returns info needed for Map from Coverage.
    #
    #     Args:
    #         feature_type (str): 'points', 'arcs', or 'polygons'
    #
    #     Returns:
    #         (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
    #     """
    #     if feature_type != 'polygons':
    #         return {}
    #     return {'Name': None, 'SS': None, 'SY': None}

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return OptionsBlock(
            [
                Checkbox('HHFORMULATION_RHS', brief='Prevents adding asymmetric terms to the flow matrix'),
                CheckboxField(
                    'DENSEREF', brief='Fluid reference density used in the equation of state', type_='float', value=0.0
                ),
                CheckboxField('DENSITY FILEOUT', brief='Save density to file', type_='str'),
            ]
        )
