"""IstData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import units_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxField, CheckboxPrintFormat


class IstData(GriddataBase):
    """Data class to hold the info from a IST package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'IST6'
        self.add_block(
            'GRIDDATA',
            ['POROSITY', 'VOLFRAC', 'ZETAIM', 'CIM', 'DECAY', 'DECAY_SORBED', 'BULK_DENSITY', 'DISTCOEF', 'SP2']
        )

    def get_units(self, array_name: str) -> str:
        """Returns the units string for the array.

        Args:
            array_name (str): The name of a array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        match array_name:
            case 'POROSITY':
                units_spec = units_util.UNITS_UNITLESS
            case 'VOLFRAC':
                units_spec = units_util.UNITS_UNITLESS
            case 'ZETAIM':
                units_spec = units_util.UNITS_TIME
            case 'CIM':
                units_spec = units_util.UNITS_CONCENTRATION
            case 'DECAY':
                d = self.options_block.get('FIRST_ORDER_DECAY')
                if d:
                    units_spec = '[1/T]'
                else:
                    units_spec = '[M/L^3/T]'
            case 'DECAY_SORBED':
                d = self.options_block.get('FIRST_ORDER_DECAY')
                if d:
                    units_spec = '[1/T]'
                else:
                    units_spec = '[M/M/T]'
            case 'BULK_DENSITY':
                units_spec = units_util.UNITS_BULK_D
            case 'DISTCOEF':
                units_spec = units_util.UNITS_DIST_C
            case 'SP2':
                units_spec = units_util.UNITS_UNITLESS
            case _:
                units_spec = ''  # This is an error
        return units_spec

    def is_int_array(self, array_name):
        """Returns True if the array is integers.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return False

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() in {'POROSITY', 'VOLFRAC', 'ZETAIM'}

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Immobile Storage and Transfer (IST) Package'

    # def map_info(self, feature_type):
    #     """Returns info needed for Map from Coverage.
    #
    #     Args:
    #         feature_type (str): 'points', 'arcs', or 'polygons'
    #
    #     Returns:
    #         (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
    #     """
    #     if feature_type != 'polygons':
    #         return {}
    #     return {'HK': Mapping(mf_name='K'), 'VK': Mapping(mf_name='K33'), 'HANI': Mapping(mf_name='K22'),
    #             'VANI': Mapping(mf_name='K33')}

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return OptionsBlock(
            [
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                CheckboxField('BUDGET FILEOUT', brief='Save budget to file', type_='str'),
                CheckboxField('BUDGETCSV FILEOUT', brief='Save budget to CSV file', type_='str'),
                Checkbox('SORPTION', brief='Sorption will be activated'),
                Checkbox('FIRST_ORDER_DECAY', brief='First order decay will occur'),
                Checkbox('ZERO_ORDER_DECAY', brief='Zero order decay will occur'),
                CheckboxField('CIM FILEOUT', brief='Save immobile concentration to file', type_='str'),
                CheckboxPrintFormat('CIM PRINT_FORMAT', brief='Format for printing to the listing file'),
                CheckboxField('SORBATE FILEOUT', brief='Save immobile sorbate concentrations', type_='str'),
            ]
        )

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        txt = ''
        if tab_text == 'POROSITY':
            txt = (
                'Porosity of the immobile domain specified as the immobile domain pore volume per immobile domain'
                ' volume.'
            )
        elif tab_text == 'VOLFRAC':
            txt = (
                'Fraction of the cell volume that consists of this immobile domain. The sum of all immobile domain'
                ' volume fractions must be less than one.'
            )
        elif tab_text == 'ZETAIM':
            txt = ('Mass transfer rate coefficient between the mobile and immobile domains, in dimensions of per time.')
        elif tab_text == 'CIM':
            txt = (
                'Initial concentration of the immobile domain in mass per length cubed. If CIM is not specified, then'
                ' it is assumed to be zero.'
            )
        elif tab_text == 'DECAY':
            txt = (
                'Rate coefficient for first or zero-order decay for the aqueous phase of the immobile domain. A'
                ' negative value indicates solute production. The dimensions of decay for first-order decay is one over'
                ' time. The dimensions of decay for zero-order decay is mass per length cubed per time. Decay will have'
                ' no effect on simulation results unless either first- or zero-order decay is specified in the options'
                ' block.'
            )
        elif tab_text == 'DECAY_SORBED':
            txt = (
                'Rate coefficient for first or zero-order decay for the sorbed phase of the immobile domain. A negative'
                ' value indicates solute production. The dimensions of decay_sorbed for first-order decay is one over'
                ' time. The dimensions of decay_sorbed for zero-order decay is mass of solute per mass of aquifer per'
                ' time. If decay_sorbed is not specified and both decay and sorption are active, then the program will'
                ' terminate with an error. decay_sorbed will have no effect on simulation results unless the SORPTION'
                ' keyword and either first- or zero-order decay are specified in the options block.'
            )
        elif tab_text == 'BULK_DENSITY':
            txt = (
                'Bulk density of this immobile domain in mass per length cubed. Bulk density is defined as the immobile'
                ' domain solid mass per volume of the immobile domain. bulk_density is not required unless the SORPTION'
                ' keyword is specified in the options block. If the SORPTION keyword is not specified in the options'
                ' block, bulk_density will have no effect on simulation results.'
            )
        elif tab_text == 'DISTCOEF':
            txt = (
                'Distribution coefficient for the equilibrium-controlled linear sorption isotherm in dimensions of'
                ' length cubed per mass. distcoef is not required unless the SORPTION keyword is specified in the'
                ' options block. If the SORPTION keyword is not specified in the options block, distcoef will have no'
                ' effect on simulation results.'
            )
        elif tab_text == 'SP2':
            txt = (
                'Exponent for the Freundlich isotherm and the sorption capacity for the Langmuir isotherm. sp2 is not'
                ' required unless the SORPTION keyword is specified in the options block and sorption is specified as'
                ' FREUNDLICH or LANGMUIR. If the SORPTION keyword is not specified in the options block, or if sorption'
                ' is specified as LINEAR, sp2 will have no effect on simulation results.'
            )
        return txt
