"""ObsData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
# import pandas as pd

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.list_package_data import ListPackageData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.file_io import io_util
from xms.mf6.gui import gui_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxField


class ObsData(ListPackageData):
    """Data class to hold the info from an OBS package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'OBS6'
        self.parent_package_ftype = 'GWF6'  # Ftype of package associated with this obs file.

    def get_column_delegate_info(self, block):
        """Returns a list of tuples of [0] column index and [1] list of strings.

        Returns:
            (tuple): tuple containing:
                - index (int): Column index.
                - strings (list of str): List of strings allowed in the column.
        """
        delegate_info = None
        return delegate_info

    # @overrides
    def get_column_info(self, block, use_aux=True):
        """Returns column names, types, and defaults.

        The columns depend on the type of package (self.parent_package_ftype) that the obs file is listed in.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        columns = {
            'OBSNAME': (object, ''),
            'OBSTYPE': (object, ''),
            'ID': (object, ''),
            'ID2': (object, ''),
        }
        return gui_util.column_info_tuple_from_dict(columns)

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info('')
        return {
            names.index('OBSNAME'): 'String of 1 to 40 nonblank characters used to identify the observation.',
            names.index('OBSTYPE'): 'String of characters used to identify the observation type',
            names.index('ID'): 'Text identifying cell where observation is located',
            names.index('ID2'): 'Text identifying cell adjacent to cell identified by ID',
        }

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        You should override this method.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        return [], {}, {}

    def dialog_title(self):
        """Returns the title to show in the dialog.

        You should override this method.

        Returns:
            (str): The dialog title.
        """
        return 'Observations (OBS) Dialog'

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return OptionsBlock(
            [
                CheckboxField(
                    'DIGITS',
                    brief='Digits used for conversion of simulated values to text on output',
                    type_='int',
                    value=5
                ),
                Checkbox('PRINT_INPUT', brief='Print input to listing file'),
            ]
        )

    # @overrides
    def read_csv_file_into_dataframe(self, block, filename, column_names, column_types):
        """Reads a csv file and returns a dataframe.

        Overridden to use comma separators.

        Args:
            block (str): Name of the block.
            filename (str): Filepath.
            column_names (list): Column names.
            types (dict of str -> type): Column names -> column types.

        Returns:
            (pandas.DataFrame)
        """
        return gui_util.read_csv_file_into_dataframe(filename, column_names, column_types, separator=',')

    # @overrides
    def dataframe_to_temp_file(self, block, dataframe):
        """Writes the dataframe to a csv file.

        Overridden because obs ID and ID2 columns can be a cellid (with up to 3 numbers) or just one string.

        Args:
            block (str): Name of the block.
            dataframe (pandas.DataFrame): The dataframe

        Returns:
            (str): Filepath of file created.
        """
        # Write to a string so we can strip the quotes before dumping to file
        # WE CAN'T SUPPORT BOUNDNAMES WITH SPACES. How would we know the difference between a boundname with
        # spaces and a 3-tuple cellid?
        csv_string = gui_util.dataframe_to_csv(dataframe, '', separator=',')
        csv_string = csv_string.replace("'", '')  # Remove quote chars

        # Can't just dump csv_string - we get too many newlines. Have to do all this.
        with open(io_util.get_temp_filename(), mode='wt') as file:
            lines = csv_string.splitlines()
            for line in lines:
                file.write(line + '\n')
            file.close()
            return file.name
