"""ZoneWriter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.file_io.package_writer_base import PackageWriterBase


class ZoneWriter(PackageWriterBase):
    """Writes the package to disk."""
    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self._fp = None
        self._sub_dir = ''

    def _write_dimensions(self, fp):
        """Writes the dimensions block."""
        fp.write('BEGIN DIMENSIONS\n')
        fp.write(f'  NCELLS {self._data.ncells}\n')
        fp.write('END DIMENSIONS\n')

    def _write_zone_name_file(self, zone_filename: str) -> None:
        """Write the .nam file.

        Args:
            zone_filename: zone .nam filepath.
        """
        model = self._data.model
        model_output_dir = f'..\\{os.path.splitext(model.mname)[0]}_output'
        budget_filename = os.path.basename(data_util.get_budget_file_name(model))
        budget_path = os.path.join(model_output_dir, budget_filename)
        zone_filename = os.path.basename(zone_filename)
        grb_filename = f'..\\{_get_grb_filename(model)}'
        zbud_name_filename = self._get_zone_budget_name_file_filename(budget_filename)

        # Add single quotes if spaces in filenames
        budget_path = f"'{budget_path}'" if budget_path.find(' ') > -1 else budget_path
        zone_filename = f"'{zone_filename}'" if zone_filename.find(' ') > -1 else zone_filename
        grb_filename = f"'{grb_filename}'" if grb_filename.find(' ') > -1 else grb_filename

        with open(zbud_name_filename, 'w') as file:
            file.write('BEGIN ZONEBUDGET\n')
            file.write(f"  BUD {budget_path}\n")
            file.write(f"  ZON {zone_filename}\n")
            file.write(f"  GRB {grb_filename}\n")
            file.write('END ZONEBUDGET\n')

    def _get_zone_budget_name_file_filename(self, budget_filename: str) -> str:
        zbud_name_file = os.path.join(self._sub_dir, f'{os.path.splitext(budget_filename)[0]}.nam')
        return zbud_name_file

    def _write_package(self, data):
        """Writes the package data to disk.

        Args:
            data (NpfData): The package data.
        """
        self._data = data
        grid_info = data.grid_info()

        # Initialize if it isn't already
        if self._data.ncells == 0:
            if data._base._grid_info:
                # data._grid_info used here to get correct grid size. See default_package_creator.create_package()
                grid_info = data._base._grid_info
            self._data.ncells = grid_info.cell_count()
        if self._data.block('GRIDDATA').array('IZONE') is None:
            self._data.init_izone(grid_info)

        self._sub_dir = ''
        if not self._writer_options.dmi_sim_dir:  # Native text
            self._sub_dir, _, _ = self._writer_options.output_dir.rpartition('_output')
            self._sub_dir += '_zone_budget'
            if not os.path.isdir(self._sub_dir):
                fs.make_or_clear_dir(self._sub_dir)
            self._write_zone_name_file(data.filename)
            old_sim_dir = self._writer_options.mfsim_dir
            self._writer_options.mfsim_dir = self._sub_dir
            old_open_close_dir = self._writer_options.open_close_dir
            self._writer_options.open_close_dir = os.path.join(self._sub_dir, os.path.basename(old_open_close_dir))
            if not os.path.isdir(self._writer_options.open_close_dir):
                fs.make_or_clear_dir(self._writer_options.open_close_dir)
            data.filename = os.path.join(self._sub_dir, os.path.basename(data.filename))

        with open(data.filename, 'w') as self._fp:
            self._write_dimensions(self._fp)
            self._write_griddata(self._fp)
            if self._sub_dir:
                self._writer_options.mfsim_dir = old_sim_dir
                self._writer_options.open_close_dir = old_open_close_dir


def _get_grb_filename(model) -> str:
    """Given a model, return the name of the .grb (binary grid file).

    Args:
        model: The model

    Returns:
        See description.
    """
    dis = model.get_dis()
    return f'{Path(dis.filename).name}.grb'
