"""DefinePeriodsDialog."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass

# 2. Third party modules
from PySide2.QtWidgets import (
    QApplication, QDialog, QDialogButtonBox, QHBoxLayout, QLabel, QSpinBox, QVBoxLayout, QWidget
)

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.mf6.gui import gui_util
from xms.mf6.gui.dialog_input import DialogInput


@dataclass
class CopyPeriodRv:
    """Results from the dialog."""
    copy_from: int
    copy_to: list[int]


def run_dialog(
    dlg_input: DialogInput, start_period: int, defined_periods: list[int], parent: QWidget | None
) -> CopyPeriodRv | None:
    """Run the dialog and return the list of defined periods, or return None if cancelled.

    Args:
        dlg_input: Information needed by the dialog.
        start_period: The current period (1-based).
        defined_periods: The list of periods already defined.
        parent: The parent window.
    """
    dialog = CopyPeriodDialog(dlg_input, start_period, defined_periods, parent)
    if dialog.exec() == QDialog.Accepted:
        return dialog.get_rv()
    return None


class CopyPeriodDialog(XmsDlg):
    """Dialog to define multiple stress periods."""

    COPY_TO_COL = 'Copy To'
    DEFINED_COL = 'Defined'

    def __init__(
        self, dlg_input: DialogInput, start_period: int, defined_periods: list[int] | None, parent: QWidget | None
    ):
        """Initializes the dialog.

        Args:
            dlg_input: Information needed by the dialog.
            start_period: The period (1-based) that the 'Copy from period' spin box is set to.
            defined_periods: The list of periods already defined.
            parent: The parent window.
        """
        super().__init__(parent, 'xms.mf6.gui.package_dialog_base')

        self._start_period = start_period
        self._defined_periods = defined_periods
        self._table = None
        self._btn_box = None
        self.help_getter = gui_util.help_getter(gui_util.help_id_from_key('AddStressesDialog'))

        self.setWindowTitle('Copy Period')
        self.setLayout(QVBoxLayout())
        self._add_period_spinbox()
        self._add_periods_label()
        self._add_table(defined_periods, dlg_input)
        self._add_buttonbox()

    def get_rv(self) -> CopyPeriodRv:
        """Return the results in a CopyPeriodRv class."""
        model = self._table.model()
        return CopyPeriodRv(self._spin_box.value(), model.data_frame[self.COPY_TO_COL].to_list())

    def _add_period_spinbox(self) -> None:
        """Add the period spinbox."""
        hlay_spn = QHBoxLayout()
        hlay_spn.addWidget(QLabel('Copy from period:'))
        self._spin_box = QSpinBox(self)
        self._spin_box.setRange(1, len(self._defined_periods))
        self._spin_box.setValue(self._start_period)
        self._spin_box.setMinimumWidth(40)
        hlay_spn.addWidget(self._spin_box)
        hlay_spn.addStretch()
        self.layout().addLayout(hlay_spn)

    def _add_periods_label(self) -> None:
        """Add the 'Periods:' label."""
        self.layout().addWidget(QLabel('Periods:'))

    def _add_table(self, defined_periods: list[int], dlg_input: DialogInput) -> None:
        """Add the table."""
        df = dlg_input.data.mfsim.tdis.period_df.copy()
        # Add two columns for 'Copy To' and 'Defined'
        df.insert(0, self.DEFINED_COL, defined_periods)
        df.insert(0, self.COPY_TO_COL, [0] * len(df))
        model = QxPandasTableModel(df)
        model.set_checkbox_columns({0, 1})
        read_only_columns = set(range(1, len(df.columns)))
        model.set_read_only_columns(read_only_columns)

        self._table = gui_util.new_table_view()
        self._table.use_header_checkboxes = True
        self._table.allow_drag_fill = True
        self._table.setModel(model)
        self.layout().addWidget(self._table)
        widget_builder.resize_columns_to_contents(self._table)
        QApplication.processEvents()
        self.adjustSize()

    def _add_buttonbox(self) -> None:
        """Add the OK/Cancel buttons."""
        w = self.btn_box = QDialogButtonBox()
        self.layout().addWidget(w)
        self.btn_box.setStandardButtons(QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help)
        self.btn_box.accepted.connect(self.accept)
        self.btn_box.rejected.connect(self.reject)
        self.btn_box.helpRequested.connect(self.help_requested)
