"""OcDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import (QComboBox, QHBoxLayout, QLabel)
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data import data_util, oc_data
from xms.mf6.file_io import io_util
from xms.mf6.gui import gui_util
from xms.mf6.gui.options_gui import OptionsGui
from xms.mf6.gui.package_dialog import PackageDialog
from xms.mf6.gui.widgets.period_list_widget import PeriodListWidget


class OcDialog(PackageDialog):
    """The Output Control (OC) package dialog."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the dialog.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'PERIODS']
        self.default_sections = ['PERIODS']

    def setup_section(self, section_name):
        """Sets up a section of widgets.

        Args:
            section_name (str): name of the section
        """
        if section_name == 'PERIODS':
            self.setup_periods_section()
        else:
            super().setup_section(section_name)

    # @overrides
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)
        self._add_auto_name_warning(vlayout)
        self._change_output_fields_if_auto_naming()

    def _add_auto_name_warning(self, vlayout):
        # GMS auto name option
        if self._auto_file_naming():
            label_str = (
                'File names are being set automatically by GMS. To disable this,'
                ' go to the Simulation Options dialog and, under "GMS Options", turn off the'
                ' "Automatically name all files..." check box.'
            )
            txt = self.options_gui.uix['txt_auto_name'] = QLabel(label_str)
            gui_util.set_label_styles_for_warning_message(txt)

            vlayout.insertWidget(0, txt)

    def _auto_file_naming(self):
        """Returns True if the Auto name checkbox is checked.

        Returns:
            (bool): See description.
        """
        return self.dlg_input.data.mfsim.gms_options.get('AUTO_FILE_NAMING', False)

    def _change_output_fields_if_auto_naming(self):
        if self._auto_file_naming():
            for widget_name, widget in self.options_gui.uix.items():
                if widget_name.startswith('edt_') and 'fileout' in widget_name:
                    gui_util.set_read_only_and_grey(widget, True)
                    extension = oc_data.fileout_extension(widget_name)
                    prefix = self.dlg_input.data.model.mname
                    widget.setText(data_util.auto_file_name(prefix, extension))

    def _add_presets_combo_box(self):
        """Adds the preset output combo box."""
        section = 'PERIODS'

        # Presets combo box
        hlayout = QHBoxLayout()
        w = self.uix[section]['txt_preset'] = QLabel('Preset output')
        hlayout.addWidget(w)
        w = self.uix[section]['cbx_preset'] = QComboBox()
        w.addItems(['At every time step', 'At last time step of each stress period', 'User specified'])
        w.setCurrentIndex(int(self.dlg_input.data.preset_output))
        hlayout.addWidget(w)
        hlayout.addStretch()
        self.uix['PERIODS']['layout'].addLayout(hlayout)

    def setup_periods_section(self):
        """Sets up the PERIODS section."""
        section = 'PERIODS'
        self.add_group_box_to_scroll_area(section)

        self._add_presets_combo_box()

        # Add stress period list widget
        w = self.uix[section]['sp_list_widget'] = PeriodListWidget(self.dlg_input.data, self.dlg_input, self)
        self.sp_widget = w
        w.actions[':/resources/icons/plot.svg'].setVisible(False)  # Hide XY Series button. Not yet implemented.
        self.uix[section]['layout'].addWidget(self.uix[section]['sp_list_widget'])

    def setup_signals(self):
        """Sets up any needed signals."""
        super().setup_signals()
        self.uix['PERIODS']['cbx_preset'].currentTextChanged.connect(self.do_enabling)

    def do_enabling(self):
        """Enables and disables widgets appropriately."""
        # Combo box and toolbar
        super().do_enabling()
        self.uix['PERIODS']['cbx_preset'].setEnabled(not self.dlg_input.locked)

        # Table
        show_table = (self.uix['PERIODS']['cbx_preset'].currentText() == 'User specified')
        self.uix['PERIODS']['sp_list_widget'].setEnabled(show_table)

    def clean_up_temp_files(self):
        """Deletes all temp files."""
        io_util.clean_up_temp_files(self.dlg_input.data.period_files.values())

    @override
    def widgets_to_data(self) -> None:
        """Set dlg_input.data from widgets."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.dlg_input.data.preset_output = self.uix['PERIODS']['cbx_preset'].currentIndex()
