"""PackageDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.gui import list_dialog
from xms.mf6.gui.obs_dialog import ObsDialog
from xms.mf6.gui.options_gui import OptionsGui
from xms.mf6.gui.package_dialog_base import PackageDialogBase
from xms.mf6.gui.widgets.period_array_widget import PeriodArrayWidget
from xms.mf6.gui.widgets.period_list_widget import PeriodListWidget


def create_object_with_base_class_copy(obj, new_type):
    """
    Creates a new object with a copy of the base class of the given object.

    Got this from AI. Use with caution.

    Args:
        obj: The object whose base class should be copied.
        new_type: The new object type (class).

    Returns:
        A new object with the same base class and attributes as the input object.
    """
    new_obj = new_type(**obj.__dict__)
    return new_obj


class PackageDialog(PackageDialogBase):
    """A dialog or base class for all list-based and array-based packages."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the simulation.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)
        self.setup_ui()

    @override
    def clear_sections(self) -> None:
        """Clear all section widgets."""
        # Close the database, if necessary
        sp_list_widget = self.uix.get('PERIODS', {}).get('sp_list_widget')
        if sp_list_widget:
            sp_list_widget.close_db()
        self.sp_widget = None
        super().clear_sections()

    def close_db(self):
        """Closes the list widget database."""
        self.uix['PERIODS']['sp_list_widget'].close_db()

    @override
    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'PERIODS']
        self.default_sections = ['PERIODS']

    @override
    def setup_section(self, section_name):
        """Sets up a section of widgets.

        Args:
            section_name (str): name of the section
        """
        if section_name == 'PERIODS':
            self.setup_periods_section()
        else:
            super().setup_section(section_name)

    @override
    def do_enabling(self):
        """Enables and disables widgets appropriately."""
        super().do_enabling()
        if self.sp_widget:
            self.sp_widget.do_enabling()
        if 'chk_readasarrays' in self.options_gui.uix:
            self.options_gui.uix['chk_readasarrays'].setEnabled(False)

    @override
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)
        if 'chk_readasarrays' in self.options_gui.uix:
            self.options_gui.uix['chk_readasarrays'].setChecked(True)

    def setup_periods_section(self):
        """Adds the stress period widget now that the file has been read."""
        name = 'PERIODS'
        self.add_group_box_to_scroll_area(name)

        if self.dlg_input.data.options_block.has('READASARRAYS'):
            # Create the array widget
            w = self.uix[name]['sp_array_widget'] = PeriodArrayWidget(self.dlg_input.data, self.dlg_input, self)
            self.uix[name]['layout'].addWidget(w)
            self.sp_widget = w
        else:
            # Create the list widget
            w = self.uix[name]['sp_list_widget'] = PeriodListWidget(self.dlg_input.data, self.dlg_input, self)
            self.uix[name]['layout'].addWidget(w)
            self.sp_widget = w

    def on_btn_obs6_filein(self):
        """Opens the dialog and updates the list of observation files."""
        list_dialog.run_list_dialog_for_filein(
            ftype='OBS6',
            parent=self,
            options_block=self.dlg_input.data.options_block,
            edit_func=ObsDialog.run_dialog_on_file
        )

    def on_chk_boundnames(self, checked):
        """Warns user that this will add or remove the boundname column and lets them cancel.

        Args:
            checked (bool): True if checkbox is checked.
        """
        if not self.loaded:
            return
        use_aux = 'chk_auxiliary' in self.options_gui.uix and self.options_gui.uix['chk_auxiliary'].isChecked()
        self.sp_widget.change_boundnames(checked == Qt.Checked, use_aux=use_aux)

    def get_data(self):
        """Returns the self.dlg_input.data object.

        Returns:
            See description.
        """
        return self.dlg_input.data

    @override
    def widgets_to_data(self) -> None:
        """Get info from widgets and store it in dlg_input.data."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.sp_widget.accept()
        sp_list_widget = self.uix.get('PERIODS', {}).get('sp_list_widget')
        if sp_list_widget:
            sp_list_widget.close_db()

    @override
    def reject(self) -> None:
        """Called when the user clicks Cancel."""
        self.sp_widget.reject()
        super().reject()
