"""XY Series functions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


def value_from_time(x, y, time):
    """Returns a value from a timeseries that corresponds to the given time.

    Interpolates if the time is not on an even increment. All lines are assumed to be constant value before the first
    and after the last point. All times must be sorted in ascending order.

    Returns:
        (tuple): tuple containing:
            - (float) The value.
            - (bool) Flag indicating we extrapolated or there were no x y values.
    """
    flag = False
    npts = len(x)

    if npts == 0:
        flag = True
        return 0.0, flag

    # Time is after max time
    if time >= x[npts - 1]:
        if time > x[npts - 1]:
            flag = True
        return y[npts - 1], flag

    # Time is before min time
    if time <= x[0]:
        if time < x[0]:
            flag = True
        return y[0], flag

    # Interpolate time
    for i in range(npts):
        x0, y0 = x[i - 1], y[i - 1]
        x1, y1 = x[i], y[i]
        if x1 >= time:
            t = (time - x0) / (x1 - x0)
            return (y1 * t) + (y0 * (1.0 - t)), flag


def average_value_from_time_range(x, y, begtime, endtime):
    """Returns the average value over the given timerange for the curve.

    See xyAverageValueFromTimeRange.

    Args:
        x (list): X values.
        y (list): Y values.
        begtime (float): Time at the beginning of the range.
        endtime (float): Time at the end of the range.

    Returns:
        (tuple): tuple containing:

            - (float): The value.
            - (bool): Flag indicating we extrapolated or there were no x y values.
            - (str): Reason why the flag is True (if it is).

                - 'no data', 'begin >= end', 'range is partially before data', 'range is completely before data',
                 'range is partially after data', 'range is completely after data'
    """
    if not x or not y:
        return -1.0, True, 'no data'

    flag = False
    reason = ''
    numpts = len(x)

    if begtime >= endtime:
        value, flag = value_from_time(x, y, (begtime + endtime) / 2.0)
        return value, flag, 'begin >= end'

    area = 0.0
    # Take care of stuff before timeseries
    if begtime <= x[0]:
        if begtime < x[0]:
            flag = True
            reason = 'partially before'
        if endtime < x[0]:
            return y[0], flag, 'completely before'
        elif endtime == x[0]:
            return y[0], flag, 'partially before'
        else:
            area += y[0] * (x[0] - begtime)

    # Take care of stuff inside timeseries
    lasttime = begtime
    lastvalue = y[0]
    firstfound = False
    for i in range(numpts - 1):
        if x[i] >= begtime:
            lasttime = x[i]
            lastvalue = y[i]
            firstfound = True
        elif x[i + 1] > begtime:
            t = (begtime - x[i]) / (x[i + 1] - x[i])
            lasttime = begtime
            lastvalue = (y[i + 1] * t) + (y[i] * (1.0 - t))
            firstfound = True

        if firstfound:
            if x[i + 1] < endtime:
                area += ((lastvalue + y[i + 1]) / 2.0) * (x[i + 1] - lasttime)
            else:
                t = (endtime - x[i]) / (x[i + 1] - x[i])
                thisvalue = (y[i + 1] * t) + (y[i] * (1.0 - t))
                area += ((lastvalue + thisvalue) / 2.0) * (endtime - lasttime)
                break

    # Take care of stuff after timeseries
    if endtime >= x[numpts - 1]:
        if endtime > x[numpts - 1]:
            flag = True
            reason = 'partially after'
        if begtime > x[numpts - 1]:
            return y[numpts - 1], flag, 'completely after'
        elif begtime == x[numpts - 1]:
            return y[numpts - 1], flag, 'partially after'
        else:
            area += y[numpts - 1] * (endtime - x[numpts - 1])

    return (area / (endtime - begtime)), flag, reason
