"""Module for exporting mesh files."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['export_mesh_files']

# 1. Standard Python modules
from pathlib import Path
import shutil
from typing import Optional

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.components.dmi.xms_data import XmsData
from xms.gmi.data.generic_model import GenericModel, Section
from xms.gmi.gui.section_dialog import SectionDialog
from xms.guipy.dialogs.feedback_thread import FeedbackThread
from xms.guipy.dialogs.process_feedback_dlg import run_feedback_dialog

# 4. Local modules


def export_mesh_files(data: XmsData, parent: QWidget):
    """
    Export the .grd, .neighbors, and/or .bc files from the simulation folder.

    Args:
        data: High-level interprocess communication object.
        parent: Parent widget for new windows created.
    """
    ok, mesh_file, neighbor_file = prompt_for_files(parent)

    if not ok:
        return

    worker = ExportMeshFilesThread(data, mesh_file, neighbor_file)
    run_feedback_dialog(worker, parent)


def model() -> Section:
    """Model for the export dialog."""
    dependencies = {True: True, False: False}
    section = GenericModel().global_parameters
    group = section.add_group('group', 'group')

    export_mesh = group.add_boolean('export_mesh', 'Export mesh', default=False)
    mesh_file = group.add_output_file('mesh_file', 'Mesh file', default='', file_filter='Mesh files (*.grd *.14)')
    mesh_file.add_dependency(export_mesh, dependencies)

    export_neighbor = group.add_boolean('export_neighbor', 'Export neighbor', default=False)
    neighbor_file = group.add_output_file(
        'neighbor_file', 'Neighbor file', default='', file_filter='Neighbor files (*.neighbors)'
    )
    neighbor_file.add_dependency(export_neighbor, dependencies)

    return section


def prompt_for_files(parent: QWidget) -> tuple[bool, Optional[Path], Optional[Path]]:
    """
    Prompt the user for whether and where to save the .grd, .neighbors and .bc files.

    Returns:
        Tuple of (ok, grd_file, neighbor_file, bc_file).

        If ok is True, the user accepted the dialog and wanted files exported. Otherwise, the user changed their mind
        and wanted to stop the process.

        If a file is None, the user did not want it exported. Otherwise, it is the path the user wants that file
        exported to.
    """
    section = model()
    dlg = SectionDialog(
        parent=parent,
        section=section,
        dlg_name='xms.ptm.feedback.export_mesh_files_thread',
        window_title='Export mesh files',
        show_groups=False,
    )
    if not dlg.exec():
        return False, None, None

    group = dlg.section.group('group')

    if group.parameter('export_mesh').value:
        mesh_file = Path(group.parameter('mesh_file').value)
    else:
        mesh_file = None

    if group.parameter('export_neighbor').value:
        neighbor_file = Path(group.parameter('neighbor_file').value)
    else:
        neighbor_file = None

    return True, mesh_file, neighbor_file


class ExportMeshFilesThread(FeedbackThread):
    """Thread for exporting .grd, .neighbors and .bc files from the simulation folder."""
    def __init__(self, data: XmsData, grid_file: Optional[Path], neighbor_file: Optional[Path]):
        """
        Construct the worker.

        Args:
            data: Interprocess communication object.
            grid_file: Where to export the .grd file to. If None, not exported.
            neighbor_file: Where to export the .neighbors file to. If None, not exported.
        """
        super().__init__(xms_data=data)
        self.display_text |= {
            'title': 'Writing geometry files...',
            'working_prompt': 'Writing geometry files. Please wait...',
        }
        self._data = data
        self.grid_file = grid_file
        self.neighbor_file = neighbor_file

    def _run(self):
        """Run the thread."""
        self.copy('.grd', self.grid_file)
        self.copy('.neighbors', self.neighbor_file)

    def copy(self, extension: str, destination: Path):
        """Copy the file with the given extension out of the model directory to the given destination."""
        if not destination:
            self._log.info(f'{extension} file not set for export, skipping...')
            return

        source = self._data.simulation_folder / f'mesh{extension}'
        if not source.exists():
            self._log.error(f'{extension} file did not exist. Export and run the simulation to generate it.')
            return

        self._log.info(f'Exporting {extension} file...')
        shutil.copy(source, destination)
