"""Module for the trap reader."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['read_traps']

# 1. Standard Python modules
from pathlib import Path
from typing import cast, TextIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.component_builders.coverage_component_builder import CoverageComponentBuilder
from xms.data_objects.parameters import Coverage
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData
from xms.guipy.data.target_type import TargetType
from xms.ptmio.trap.trap_reader import read_traps as pio_read
from xms.ptmio.trap.traps import Direction, LineTrap, PolygonTrap

# 4. Local modules
from xms.ptm.components.traps_component import PtmTrapsComponent
from xms.ptm.model.model import trap_model


def read_traps(where: str | Path | TextIO) -> tuple[Coverage, PtmTrapsComponent]:
    """
    Read traps out of a trap file.

    Args:
        where: The file to read.

    Returns:
        A coverage and component with the contents of the file.
    """
    traps = pio_read(where)

    builder = CoverageComponentBuilder(PtmTrapsComponent, 'PTM Traps')

    build_lines(traps.line_traps, builder)
    build_polygons(traps.polygon_traps, builder)

    coverage, component = builder.build(using_xms_data=True)
    return coverage, component


def build_lines(traps: list[LineTrap], builder: CoverageComponentBuilder):
    """Add line traps to the coverage."""
    data = cast(CoverageBaseData, builder.data)
    section = trap_model().arc_parameters
    group = section.group('horizontal_line')
    mapping = {
        Direction.decreasing_x: 'Decreasing x-coordinate',
        Direction.increasing_x: 'Increasing x-coordinate',
        Direction.any: 'Either Direction',
    }

    for trap in traps:
        group.parameter('id').value = trap.trap_id
        group.parameter('name').value = trap.label
        group.parameter('direction').value = mapping[trap.direction]
        group.parameter('bottom').value = trap.bottom
        group.parameter('top').value = trap.top
        group.parameter('is_open').value = trap.is_open
        group.parameter('is_single').value = trap.is_single

        component_id = data.add_feature(TargetType.arc, section.extract_values(), 'horizontal_line')

        start = (trap.start[0], trap.start[1], trap.bottom)
        end = (trap.end[0], trap.end[1], trap.bottom)
        line = [start, end]
        builder.add_point_string(line, component_id=component_id)


def build_polygons(traps: list[PolygonTrap], builder: CoverageComponentBuilder):
    """Add polygon traps to the coverage."""
    data = cast(CoverageBaseData, builder.data)
    section = trap_model().polygon_parameters
    group = section.group('polygon')

    for trap in traps:
        group.parameter('id').value = trap.trap_id
        group.parameter('name').value = trap.label
        group.parameter('bottom').value = trap.bottom
        group.parameter('top').value = trap.top
        group.parameter('is_open').value = trap.is_open
        group.parameter('is_single').value = trap.is_single

        component_id = data.add_feature(TargetType.polygon, section.extract_values(), 'polygon')

        points = [[(point[0], point[1], trap.bottom) for point in trap.points]]
        builder.add_polygon(points, component_id=component_id)
