"""Writer for fort.14 dataset files."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['write_datasets']

# 1. Standard Python modules
from pathlib import Path
from typing import Optional, Sequence

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.grid.ugrid import UGrid

# 4. Local modules


def write_datasets(datasets: list[Sequence[float]], path: Path | str, ugrid: Optional[UGrid] = None):
    """
    Write a .fort.14 dataset file.

    Values are assigned IDs starting from 1 and incrementing by 1. If a geometry and dataset file will both be written,
    it is the caller's responsibility to ensure the values in the datasets are in the same order as the locations in
    the geometry.

    Args:
        datasets: The datasets to write, in the order they should be written. The first dataset will be written in the
            first column, the second in the second, etc.
        path: The path to write the file to.
        ugrid: UGrid containing the point locations to write. If omitted, all locations are written as 0s.
    """
    if not datasets:
        raise AssertionError('datasets is empty.')

    length = len(datasets[0])
    for dataset in datasets:
        if len(dataset) != length:
            raise AssertionError('Datasets have unequal lengths.')

    if ugrid and ugrid.point_count != length:
        raise AssertionError('Datasets have different length from UGrid point list.')

    ids = (i for i in range(1, length + 1))
    if ugrid:
        xs, ys = ugrid.locations[:, [0, 1]].transpose()
    else:
        xs, ys = np.zeros_like([], shape=(2, length))

    with open(path, 'w') as f:
        f.write('\n')
        f.write(f'0 {length}\n')

        for node_id, x, y, *values in zip(ids, xs, ys, *datasets):
            f.write(f'{node_id} {x} {y} ')
            f.write(' '.join(str(value) for value in values))
            f.write('\n')
