"""Module for the ProgramControl class and related enumerations."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = [
    'AdvectionMethod', 'BottomFlowFormat', 'CentroidMethod', 'EulerianMethod', 'EulerianTransportMethod', 'FlowFormat',
    'MeshFormat', 'NumericalScheme', 'ProgramControl', 'SedimentFormat', 'VelocityMethod', 'WaveFormat'
]

# 1. Standard Python modules
from dataclasses import dataclass
from datetime import datetime
from enum import auto, Enum
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class EulerianMethod(Enum):
    """
    Method to use to determine the native sediment mobility and shears on the Eulerian grid.

    `ptm` is much faster.
    """
    ptm = auto()
    van_rijn = auto()


class EulerianTransportMethod(Enum):
    """The Eulerian transport method."""
    soulsby_van_rijn = auto()
    van_rijn = auto()
    lund = auto()
    camenen_larson = auto()


class CentroidMethod(Enum):
    """
    Method used to determine the parcel mobility and shears.

    The rouse method uses empirical approximations of integrated Rouse curves. The van_rijn method uses a numerical
    integration of the total sediment transport at the location of the parcel to compute the centroid elevation. The
    rouse method is much faster.
    """
    rouse = auto()
    van_rijn = auto()


class AdvectionMethod(Enum):
    """
    Method used for the advection of parcels.

    - The one_d method places the parcel at the elevation of the centroid of the local total load distribution.
    - The two_d method allows the parcel to be above or below the total load centroid elevation with the parcel moving
      vertically towards the centroid.
    - The 3d option allows the parcel to move freely in the vertical in response to a vertical force balance.
    - TODO: Document the q_three_d option.
    """
    one_d = auto()
    two_d = auto()
    three_d = auto()
    q_three_d = auto()


class VelocityMethod(Enum):
    """
    Method used to compute the vertical velocity profile.

    - two_d_log causes the model to use a logarithmic velocity profile.
    - two_d_uniform causes the model to use the given velocity throughout the water column.
      This method is designed primarily for testing.
    - two_d_two_point doesn't have any documentation. # TODO: Find documentation.
    - three_d_sigma  TODO: Find documentation.
    - three_d_z  TODO: Find documentation
    """
    two_d_log = auto()  # 2d (log), 2d (logarithmic), logarithmic,
    two_d_two_point = auto()  # two-point, two_point, 2d (two-point), 2d (two_point),
    two_d_uniform = auto()  # 2d (uniform), uniform,
    three_d_sigma = auto()  # 3d, 3ds, fully-3d
    three_d_z = auto()  # 3dz


class MeshFormat(Enum):
    """
    The format of the mesh file.

    - adcirc uses the fort.14 file format used by ADCIRC.
    - cms_2d  TODO: Find documentation
    - cms_3d  TODO: Find documentation
    - ch_3d  TODO: Find documentation
    """
    adcirc = auto()
    # m2d = auto()  No longer supported, rewritten to cms_2d
    cms_2d = auto()  # cms-m2d, cms-2d


class FlowFormat(Enum):
    """
    The format of the hydrodynamic files.

    - adcirc   TODO: Find documentation
    - cms_2d  TODO: Find documentation
    - cms_3d  TODO: Find documentation
    - ch_3d  TODO: Find documentation
    """
    adcirc_ascii = auto()  # adcirc
    adcirc_xmdf = auto()  # xmdf
    cmsflow_2d_single = auto()  # cms-2d, cms-m2d, m2d
    cmsflow_2d_multi = auto()  # cms-2d-multi
    adh = auto()  # adh


class WaveFormat(Enum):
    """
    The format of the wave file.

    TODO: Find out what the formats are.
    """
    stwave = auto()  # stwave
    wabed = auto()  # wabed
    xmdf = auto()  # xmdf
    cms_wave = auto()  # cms-wave


class SedimentFormat(Enum):
    """
    Format of the native sediment file.

    TODO: Find out what the formats are.
    """
    adcirc = auto()  # adcirc
    m2d = auto()  # cms-2d, m2d
    xmdf_dataset = auto()  # xmdf, xmdf_dataset, xmdf_property
    uniform = auto()  # uniform


class BottomFlowFormat(Enum):
    """
    Format of the bottom currents file.

    TODO: Find out what the formats are.
    A comment in XMS code suggests we just ignore this and figure it out based on the file's extension.
    """
    adcirc = auto()
    xmdf = auto()


class NumericalScheme(Enum):
    """
    The order of the numerical scheme.

    TODO: Find out why they're just numbers. Is there a more meaningful name for these?
    """
    two = auto()
    four = auto()


default_date = datetime(year=1, month=1, day=1, hour=0, minute=0, second=0)


@dataclass(slots=True)
class ProgramControl:
    """A dataclass representing a program control file."""
    #: Instructs the model that waves are to be run in the model. Turns on the various wave flags.
    # waves: bool = False
    #: Instructs the model that currents are to be run in the model. Turns on the various wave flags.
    currents: bool = False
    #: Instructs the model to only perform Eulerian calculations (i.e. no particles are released). Source file is not
    #: required if this option is used.
    # parcels: bool = True
    #: Tells the model that the input bathymetry file uses the CHC EnSim convention of positive upwards.
    # ensim: bool = False
    #: Instructs the model to output the parcel path files.
    paths: bool = False
    #: Instructs the model to output the predicted bed evolution on the Eulerian mesh node locations.
    # Frequency of output is given by `mapping_inc`. Has same effect as `bed_level_mapping`.
    morphology: bool = False
    #: Instructs the model to output the flow conditions on the Eulerian mesh node locations.
    # Frequency of output is given by `mapping_inc`.
    flow_mapping: bool = False
    # Instructs the model to output the predicted native sediment bedforms on the Eulerian mesh node locations.
    # Frequency of output is given by `mapping_inc`.
    bedform_mapping: bool = False
    # Instructs the model to output the three predicted wave parameters (H, T, θ) on the Eulerian mesh node locations.
    # Frequency of output is given by `mapping_inc`.
    wave_mapping: bool = False
    # Instructs the model to output the predicted mobility of the native sediments on the Eulerian mesh node locations.
    # Frequency of output is given by `mapping_inc`. Same result as `morphology`.
    mobility_mapping: bool = False
    # Instructs the model to output the predicted bed evolution on the Eulerian mesh node locations. Frequency of output
    # is given by `mapping_inc`. Same result as `morphology`.
    bed_level_mapping: bool = False
    # Instructs the model to output the predicted bed level change on the Eulerian mesh node locations. Frequency of
    # output is given by `mapping_inc`.
    bed_level_change_mapping: bool = False
    # Instructs the model to output the predicted potential sediment transport rate of the native sediments on the
    # Eulerian mesh node locations. Frequency of output is given by `mapping_inc`.
    transport_mapping: bool = False
    # Instructs the model to output the grain size of the parcels. Frequency of output is given by ‘output_inc’.
    grain_size_output: bool = False
    # Instructs the model to output the mobility of each parcel. Frequency of output is given by ‘output_inc’.
    mobility_output: bool = False
    # Instructs the model to output the elevation of each parcel. Frequency of output is given by ‘output_inc’.
    # elevation_output: bool = False  # Hard-coded to True since XMS doesn't support anything else.
    # Instructs the model to output the height of each parcel above the bed. Frequency of output is given by
    # ‘output_inc’.
    height_output: bool = False
    # Instructs the model to output the state of each parcel (active=1, dormant=0). Frequency of output is given by
    # ‘output_inc’.
    state_output: bool = False
    # Instructs the model to output the identification number of each parcel. Frequency of output is given by
    # ‘output_inc’.
    # id_output: bool = True  # Hard-coded to True since that's all XMS supports.
    # Instructs the model to output the original source of each parcel. Frequency of output is given by
    # ‘output_inc’.
    source_output: bool = False
    # Instructs the model to output the velocity components of each parcel. Frequency of output is given by
    # ‘output_inc’.
    flow_output: bool = False
    # Instructs the model to output the density of each parcel. Frequency of output is given by ‘output_inc’.
    density_output: bool = False
    # Instructs the model to output the mass of each parcel. Frequency of output is given by ‘output_inc’.
    parcel_mass_output: bool = False
    # The time step of the simulation in seconds.
    time_step: float = 2.0
    # Instructs the model to build up the distributions of parcel grain sizes at each source such that they are
    # Gaussian in terms of weight (i.e. “percentage finer by weight”), rather than by grain size.
    by_weight: bool = False
    # Instructs the model to skip bedform calculations on the Eulerian grid.
    bedforms: bool = True
    # TODO: Find documentation
    shear_stress_mapping: bool = False
    # Input statement for bed porosity.
    bed_porosity: float = 0.4
    # Selects which method to use to determine the native sediment mobility and shears on the Eulerian grid.
    eulerian_method: EulerianMethod = EulerianMethod.ptm
    # Selects which sediment transport method to use.
    eulerian_transport_method: EulerianTransportMethod = EulerianTransportMethod.soulsby_van_rijn
    # Selects which method to use to determine the parcel mobility and shears.
    centroid_method: CentroidMethod = CentroidMethod.rouse
    # Selects which method to use for the advection of the parcels.
    advection_method: AdvectionMethod = AdvectionMethod.two_d
    # Selects which method to use to compute the velocity profile in the vertical.
    velocity_method: VelocityMethod = VelocityMethod.two_d_log
    # Gives the format of the mesh file.
    mesh_format: MeshFormat = MeshFormat.adcirc
    # Gives the name of the mesh file.
    mesh_file: str = ''
    # Gives the format of the hydrodynamic files.
    flow_format: FlowFormat = FlowFormat.adcirc_ascii
    # This is only here for reference and should not be implemented without a good reason why it's necessary.
    # A tuple of `(flow_file_z, flow_file_uv)`. The reader internally maps the keyword for this parameter to the other
    # two instead, and the writer doesn't write them since this keyword doesn't support paths with spaces in them.
    # flow_files: tuple[str, str] = ('', '')
    # Gives name of the hydrodynamic flow file. Should typically have a .64 extension.
    # flow_file_uv: str = ''
    # Gives names of the hydrodynamic elevation file. Should typically have a .63 extension.
    # flow_file_z: str = ''
    # Gives the name of the XMDF format flow file. This is a single file in h5 format.
    flow_file_xmdf: str = ''
    # Gives name of the density/salinity/temperature file. Should typically have a .45 extension.
    # Documentation says this is not needed if IDEN=0, but I don't know where IDEN is specified. # TODO: Find out.
    # flow_file_dst: str = ''
    # Gives the name of the boundary condition file. Format depends on model type.
    bc_file: str = ''
    # Gives the internal data structure path in the XMDF format flow file.
    xmdf_wse_path: str = 'Datasets/Water Surface Elevation (63)'
    # Gives the internal data structure path in the XMDF format flow file.
    xmdf_vel_path: str = 'Datasets/Velocity (64)'
    # Gives the internal data structure path for the U velocity component in the XMDF format flow file (M3D uses three
    # scalars for the three components of velocity).
    # xmdf_u_path: str = 'Datasets/U'
    # Gives the internal data structure path for the V velocity component in the XMDF format flow file (M3D uses three
    # scalars for the three components of velocity).
    # xmdf_v_path: str = 'Datasets/U'
    # Gives the internal data structure path for the W velocity component in the XMDF format flow file (M3D uses three
    # scalars for the three components of velocity).
    # xmdf_w_path: str = 'Datasets/U'
    # Gives the name of the M2D control file {*.m2c}. This is only used for M2D version 3.02 and requires the mesh
    # format to be specified as ‘m2d’, rather than ‘cms-2d’
    # m2d_control_file: str = ''
    # Gives the internal data structure path for the depth in the XMDF format file (M3D only).
    # xmdf_dep_path: str = 'Datasets/Depth'
    # Gives the internal data structure path for the salinity in the XMDF format file (M3D and CH3D).
    # xmdf_sal_path: str = 'Datasets/Salinity'
    # Gives the internal data structure path for the temperature in the XMDF format file (M3D and CH3D).
    # xmdf_temp_path: str = 'Datasets/Temperature'
    # Gives the internal data structure path for the 3D data in the XMDF format file (M3D only).
    # xmdf_m3d_path: str = ''
    # Gives the internal data structure path for the D35 values in the XMDF format sediment file.
    xmdf_d35_path: str = ''
    # Gives the internal data structure path for the D50 values in the XMDF format sediment file.
    xmdf_d50_path: str = ''
    # Gives the internal data structure path for the D90 values in the XMDF format sediment file.
    xmdf_d90_path: str = ''
    # Time origin of the ADCIRC flow files. Each output step of the ADCIRC .63 and .64 files has a timestamp in seconds
    # that is relative to this time.
    start_flow: Optional[datetime] = None
    # Gives the format of the wave file.
    # wave_format: WaveFormat = WaveFormat.stwave
    # Gives the number of STWAVE .wav files that are to be used in the present simulation.
    # wave_files: int = 0
    # Gives the name of the XMDF input file with wave data.
    # wave_file_xmdf: str = ''
    # Gives the name of the XMDF input file with wave data.
    # wave_file_2_xmdf: str = ''
    # Gives the number of output frames (steps) that appear in each STWAVE .wav file.
    # In general, STWAVE files will contain a single frame if produced using the SMS steering module and more than one
    # frame if produced using a stand-alone version of STWAVE (e.g. steering module run -> 100 files with one frame
    # each; stand-alone -> 1 file with 100 frames). Not required if waves are unused.
    # wave_frames: int = 0
    # Gives the time (in seconds) between each output frame (step) in the STWAVE .wav file(s). Not required if waves
    # are unused.
    wave_step: float = 0.0
    # Gives the orientation angle (in degrees, clockwise from y-axis to N) of the STWAVE grid. Not required if waves
    # are unused.
    # wave_grid_angle: float = 0.0
    # x-coordinate of the STWAVE grid origin. Not required if waves are unused.
    # wave_x_origin: float = 0.0
    # y-coordinate of the STWAVE grid origin. Not required if waves are unused.
    # wave_y_origin: float = 0.0
    # Instructs the model that nested STWAVE grids are to be used. The list of file names should start on the line
    # immediately after the list of parent STWAVE file names. The nested and parent STWAVE grids must have the same
    # number of frames and the same times associated with each frame.
    # nested: bool = False
    # Gives the orientation angle (in degrees, clockwise from y-axis to N) of the nested STWAVE grid. Not required if
    # waves are unused. This is similar to `wave_grid_angle`, but this one applies to the nested grid, while that one
    # applies to the main grid.
    # wave_grid_angle_2: float = 0.0
    # x-coordinate of the nested STWAVE grid origin. Not required if
    # waves are unused. This is similar to `wave_x_origin`, but this one applies to the nested grid, while that one
    # applies to the main grid.
    # wave_x_origin_2: float = 0.0
    # y-coordinate of the nested STWAVE grid origin. Not required if
    # waves are unused. This is similar to `wave_y_origin`, but this one applies to the nested grid, while that one
    # applies to the main grid.
    # wave_y_origin_2: float = 0.0
    # Time of the first STWAVE frame. Not required if waves are unused.
    start_waves: datetime = default_date
    # Instructs model to hold waves steady, if simulation exceeds duration of wave input file.
    # exceed_waves: bool = False
    # Gives the name of the file containing the sediment source information. Generally has a .source extension.
    source_file: str = ''
    # Gives the name of the file containing the neighbor source information. Generally has a .neighbor extension.
    neighbor_file: str = ''
    # Gives the prefix of the files used for output. The model will append an extension depending on the nature of the
    # output file.
    output_prefix: str = ''
    # Gives the name of the trap file. Generally has a .dat extension. If blank, then no trap is present.
    trap_file: str = ''
    # Path to the main folder in an M2D h5 file. This is the folder with such data as the Origin, Bearing etc.
    # The datasets folder is usually below this one.
    # xmdf_grid_path: str = ''
    # Surveys all particles on the last time step and places them in a trap, whether they are moving or not.
    last_step_trap: bool = False
    # Time at which the trap, if specified, is operational. Required if trap is used.
    start_trap: datetime = default_date
    # Time at which the trap, if specified, ceases operation. Required if trap is used.
    stop_trap: datetime = default_date
    sediment_format: SedimentFormat = SedimentFormat.adcirc
    # Gives the name of the file containing the grain sizes of the native sediments
    sediment_file: str = ''
    # Start time of the simulation.
    start_run: datetime = default_date
    # Stop time of the simulation.
    stop_run: Optional[datetime] = None
    # The duration of the simulation in seconds. Must specify either :duration or :stop_time.
    duration: float = 0.0
    # The increment in steps between updating of the mobility, shears and bedforms of the Eulerian grid
    grid_update: int = 0
    # The increment in steps between updating of the flows and elevations of the Eulerian grid.
    flow_update: int = 0
    # Diffusion scalar.
    ket: float = 0.25
    # The water temperature in °C.
    temperature: float = 15.0
    # The water salinity in ‰.
    salinity: float = 34.0
    # The increment in number of steps between output of the Lagrangian parcel data.
    output_inc: int = 10
    # The increment in number of steps between output of the Eulerian grid data.
    mapping_inc: int = 1000
    # The minimum computation depth in m.
    min_depth: float = 0.01
    # Tells the model to turn off certain random generators routines and output stage times.
    # debug: bool = False
    # The model outputs a file with particle data in EnSim format (.pcl).
    # ensim_parcels: bool = False
    # The model outputs a file with map data in EnSim format (.t3s).
    # ensim_maps: bool = False
    # The model outputs a file with particle data in Tecplot format (.plt).
    tecplot_parcels: bool = False
    # The model outputs a file with map data in Tecplot format (.plt).
    tecplot_maps: bool = False
    # The model will not output a map data file in XMDF format (.h5).
    population_record: bool = False
    # Instructs the model to output the critical shear for the initiation of motion of each parcel.
    # Frequency of output is given by `output_inc`.
    tau_cr_output: bool = False
    # Instructs the model to output the fall velocity of each parcel. Frequency of output is given by `output_inc`.
    fall_velocity_output: bool = False
    # Instructs the model to use compressed XMDF files.
    xmdf_compressed: bool = False
    # Turns off hiding and exposure routines.
    hiding_exposure: bool = True
    # Turns off probabilistic shear calculations.
    turbulent_shear: bool = True
    # Turns off bed interaction routines.
    bed_interaction: bool = True
    # Turns off wave mass transport calculations.
    wave_mass_transport: bool = False
    # Model performs residency calculations (requires a polygon trap to be active)
    residence_calc: bool = False
    # Model assumes all particles are neutrally buoyant (i.e. particle density set to that of the fluid and fall
    # velocity is set to 0).
    neutrally_buoyant: bool = False
    # Gives the format of the bottom currents file.
    # bottom_flow_format: BottomFlowFormat = BottomFlowFormat.adcirc
    # Gives the internal data structure path in the XMDF format flow file.
    # xmdf_bot_path: str = 'Datasets/Velocity (64)'
    # Gives the name of the bottom currents file.
    # bottom_flow_file: str = ''
    # Sets the height of the input bottom currents.
    # bottom_flow_height: float = 0.0
    # Gives the locations of the nodes where bottom currents are provided (1) and not provided (0)
    # bottom_mask_file: str = ''
    # Time origin of the bottom flow files. Each output step of the bottom flow file has a timestamp in seconds that is
    # relative to this time.
    # bottom_start_flow: datetime = default_date
    # Instructs model to open a *.brk file for each *.wav file opened and to use breaking indices (Bi)
    # estimate increased surf zone mixing.
    # wave_breaking: bool = False
    # Sets diffusion coefficient for waves. The horizontal eddy viscosity is adjusted by the factor:
    kew: float = 5.0
    # Sets vertical diffusion coefficient.
    kev: float = 0.00859
    # Sets minimum vertical eddy viscosity.
    etmin: float = 0.02
    # Sets minimum vertical eddy viscosity.
    evmin: float = 0.0
    # Sets the z-value of parcels released by sources to a z-value relative to datum rather than the bed
    source_to_datum: bool = False
    # Controls the order of the numerical scheme. Only valid options are 2 or 4.
    numerical_scheme: NumericalScheme = NumericalScheme.two
    # Sets the density of the bed sediments. Default is 2650.
    rhos: float = 2650.0

    # UUID that PTM should put in the particle file when run
    particle_set_uuid: str = ''
