"""CanalComponent class."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.component_bases.coverage_component_base import CoverageComponentBase
from xms.gmi.data.generic_model import Section
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.rsm.dmi.canal_component_helper import CanalComponentHelper, helper_generic_model


class CanalComponent(CoverageComponentBase):
    """A Dynamic Model Interface (DMI) component base class."""
    def __init__(self, main_file: Optional[str | Path]):
        """
        Initialize the component.

        Args:
            main_file: The component's main-file.
        """
        super().__init__(main_file)
        self.tree_commands.append(('Default Canal Properties...', self.open_default_canal))
        self.canal_helper = CanalComponentHelper(self)  # do this last

    def _get_section(self, target: TargetType) -> Section:
        """Get a Section with parameters for a given target."""
        return helper_generic_model(self).section_from_target_type(target)

    def open_default_canal(self, query, params, win_cont):
        """Opens the Assign arc dialog and saves component data state on OK.

        Args:
            query (Query): Object for communicating with XMS
            params (list[dict]): Generic map of parameters. Contains selection map and component id files.
            win_cont (QWidget): The window container.

        Returns:
            (tuple(list,list)): tuple containing:

                messages (list[tuple(str,str)]): List of tuples with the first element of the tuple being the
                message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (list[ActionRequest]): List of actions for XMS to perform.
        """
        return self.canal_helper.open_default_canal(query, params, win_cont)
