"""SimComponent class."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util, TreeNode
from xms.gmi.component_bases.sim_component_base import SimComponentBase
from xms.gmi.data.generic_model import Parameter, Section
from xms.gmi.gui.dataset_callback import DatasetRequest

# 4. Local modules
from xms.rsm.data.sim_data_def import generic_model
from xms.rsm.dmi.dataset_tree import cell_dataset_selection_tree
from xms.rsm.feedback.bc_snap_runner import generate_snap


class SimComponent(SimComponentBase):
    """A hidden Dynamic Model Interface (DMI) component for the RSM model simulation."""
    def __init__(self, main_file: Optional[str | Path] = None):
        """Initializes the component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self._section_dialog_keyword_args['enable_unchecked_groups'] = True
        self._section_dialog_keyword_args['hide_checkboxes'] = True
        self.tree_commands.insert(-1, ('Generate Snap Preview...', self.generate_snap_preview))

    def _get_global_parameter_section(self) -> Section:
        """Get the global parameter section."""
        return generic_model().global_parameters

    def _dataset_callback(self, request: DatasetRequest, parameter: Parameter) -> str | TreeNode:
        """
        Handle a request for information when picking a dataset.

        This should match `xms.gmi.gui.dataset_callback.DatasetCallback`.

        Notes:
            - `self._query` will be initialized and available before this method is called.
            - If `request` is `DatasetRequest.GetTree`, the returned node will be a clean copy of the entire project
              tree. Derived classes can safely mutate it for filtering purposes.

        Args:
            request: The requested operation.
            parameter: The parameter the request is for. If `request` is `DatasetRequest.GetLabel`, then
                `parameter.value` will be initialized to the UUID of the dataset to get the label for.

        Returns:
            - If `request` is `DatasetRequest.GetLabel`, returns a label to identify the dataset with to the user.
            - If `request` is `DatasetRequest.GetTree`, returns a tree for picking a new dataset for the parameter.
        """
        if self._project_tree is None:
            self._project_tree = cell_dataset_selection_tree(self._query, self)
        if request == DatasetRequest.GetLabel:
            return tree_util.build_tree_path(self._project_tree, parameter.value)
        if request == DatasetRequest.GetTree:
            return self._project_tree
        raise AssertionError('Unknown request.')

    def generate_snap_preview(self, query, params, win_cont):
        """Creates a snapped bc component.

        Args:
            query (Query): Object for communicating with XMS
            params (list[dict]): Generic map of parameters. Contains selection map and component id files.
            win_cont (QWidget): The window container.

        Returns:
            (tuple(list,list)):
                messages (list[tuple(str,str)]): List of tuples with the first element of the tuple being the
                message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (list[ActionRequest]): List of actions for XMS to perform.
        """
        generate_snap(parent=win_cont, sim=self, query=query)
