"""canal data definition methods."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel

# 4. Local modules
import xms.rsm.data.bc_val_data_def as bc_val_dd
import xms.rsm.data.monitor_data_def as mdd

BC_NONE = 'No boundary condition'
BC_INFLOW = 'Inflow (<segmentsource>)'
BC_SPEC_HEAD = 'Specified head (<segmenthead>)'
BC_GEN_HEAD = 'General head (<segmentghb>)'
BC_SPEC_HEAD_JUNC = 'Specified head at junction (<junction head>)'
BC_UNIFORM_FLOW = 'Uniform flow (<uniformflow>)'
BC_JUNC_BLOCK = 'Structure at junction (<junctionblock>)'


def generic_model():
    """Gets a generic model for the boundary conditions coverage.

    Returns:
        (xms.gmi.data.generic_model.GenericModel): the generic model class
    """
    gm = GenericModel()
    ap = gm.arc_parameters
    # add default_canal group but no parameters.
    ap.add_group('default_canal', label='Default Canal', is_default=True)
    ap.group('default_canal').is_active = True
    _add_canal_group(ap, 'specified_canal', 'Canal')
    _add_monitor_groups(ap)
    return gm


def generic_model_default_canal():
    """Gets a generic model for the default canal.

    Returns:
        (xms.gmi.data.generic_model.GenericModel): the generic model class
    """
    gm = generic_model()
    _add_canal_group(gm.arc_parameters, 'default_canal', 'Default Canal')
    gm.arc_parameters.group('default_canal').is_active = True
    return gm


def _add_canal_group(ap, name, label):
    """Add a canal group to the arc_properties.

    Args:
        ap (GroupSet): the arc properties.
        name (str): name of group
        label (str): label to show in the gui
    """
    is_default = name == 'default_canal'
    gp = ap.add_group(group_name=name, label=label, is_default=is_default)
    gp.add_text(name='label', label='Canal name')
    gp.add_float(name='bottom_width', label='Canal bottom width', default=0.0)
    gp.add_float(name='bottom_elev', label='Canal bottom elevation', default=0.0)
    gp.add_float(name='side_slope', label='Canal side slope', default=0.0)

    flags = {True: True, False: False}
    parent = gp.add_boolean(name='b_specify_index', label='Specify index', default=False)
    p = gp.add_integer(name='index', label='', default=1, low=1)
    p.add_dependency(parent=parent, flags=flags)

    parent = gp.add_boolean(name='b_mannings', label='Mannings n', default=False)
    p = gp.add_float(name='mannings', label='Canal mannings n', default=0.0)
    p.add_dependency(parent=parent, flags=flags)

    parent = gp.add_boolean(name='b_init_head', label='Canal initial head (<initial>)', default=False)
    p = gp.add_float(name='init_head', label='', default=0.0)
    p.add_dependency(parent=parent, flags=flags)

    parent = gp.add_boolean(name='b_leakage_coeff', label='Stream-aquifer interaction (<leakage_coeff>)', default=False)
    p = gp.add_float(name='leakage_coeff', label='', default=0.0)
    p.add_dependency(parent=parent, flags=flags)

    parent = gp.add_boolean(name='b_bank', label='Stream-overland flow interaction', default=False)
    p = gp.add_float(name='bank_height', label='Canal "lip height" (<bank_height>)', default=0.0)
    p.add_dependency(parent=parent, flags=flags)
    label = 'Canal coefficient for flow over the lip to the segment (<bank_coeff>)'
    p = gp.add_float(name='bank_coeff', label=label, default=0.0)
    p.add_dependency(parent=parent, flags=flags)

    label = 'Seepage between canal and cell through a levee'
    parent = gp.add_boolean(name='b_levee', label=label, default=False)
    p = gp.add_float(name='levee', label='', default=0.0)
    p.add_dependency(parent=parent, flags=flags)

    # label = 'Seepage from a wetland cell through a levee into the canal (<levseep1>)'
    # parent = gp.add_boolean(name='b_levseep1', label=label, default=False)
    # p = gp.add_float(name='levseep1', label='', default=0.0)
    # p.add_dependency(parent=parent, flags=flags)
    #
    # label = 'Seepage from the canal into the canal (<levseep2>)'
    # parent = gp.add_boolean(name='b_levseep2', label=label, default=False)
    # p = gp.add_float(name='levseep2', label='', default=0.0)
    # p.add_dependency(parent=parent, flags=flags)

    # only add bc to the canal group and not the "default canal" group
    if name == 'specified_canal':
        _add_network_bc(gp)

    parent = gp.add_boolean(name='b_flowtype', label='Flowtype (<flowtype>)', default=False)
    opts = ['0 - Normal flow', '1 - Flow head assigned to end of canal', '2 - Segment length is very small']
    p = gp.add_option(name='flowtype', label='', default=opts[0], options=opts)
    p.add_dependency(parent=parent, flags=flags)


def _add_network_bc(gp):
    bc_type_opts = [BC_NONE, BC_INFLOW, BC_SPEC_HEAD, BC_GEN_HEAD, BC_SPEC_HEAD_JUNC]
    plabel = 'Boundary condition type (<networkbc>)'
    bc_type_par = gp.add_option(name='network_bc', label=plabel, default=bc_type_opts[0], options=bc_type_opts)

    flags_bc_type = {opt: True for opt in bc_type_opts}
    flags_bc_type[BC_NONE] = False
    p = gp.add_text(name='network_bc_label', label='Boundary condition label (<label>)')
    p.add_dependency(parent=bc_type_par, flags=flags_bc_type)
    p = gp.add_integer(name='network_bc_id', label='Boundary condition id (<bcID>)', default=-1)
    p.add_dependency(parent=bc_type_par, flags=flags_bc_type)

    flags_bc_type_ghb = {opt: False for opt in bc_type_opts}
    flags_bc_type_ghb[BC_GEN_HEAD] = True
    p = gp.add_float(name='network_bc_ghb_coeff', label='GHB coefficient (<kcoeff>)', default=0.0)
    p.add_dependency(parent=bc_type_par, flags=flags_bc_type_ghb)

    bc_val_dd.add_bc_val_to_group(gp)
    p = gp.parameter('bc_val_type')
    flags_bc_type2 = {opt: True for opt in bc_type_opts}
    flags_bc_type2[BC_NONE] = False
    p.add_dependency(parent=bc_type_par, flags=flags_bc_type2)


def _add_monitor_groups(ap):
    """Create parameters for points in the GenericModel class.

    Args:
        ap (GroupSet): the arc properties.
    """
    outputs = [
        ('head', 'Monitor head (<head>)'),
        ('head_iter', 'Monitor head at each iteration (<head_iter>)'),
        ('segmenthead', 'Monitor segment head (<segmenthead>)'),
        ('depth', 'Monitor depth (<depth>)'),
        ('sbflow', 'Monitor sum of all streambank flow in segment (<sbflow>)'),
        ('seepageflow', 'Monitor streambank flow - aquifer seepage (<seepageflow>)'),
        ('overbankflow', 'Monitor streambank flow - overbank flow (<overbankflow>)'),
        ('lev1flow', 'Monitor streambank flow - type 1 levee seepage (<lev1flow>)'),
        ('lev2flow', 'Monitor streambank flow - type 2 levee seepage (<lev2flow>)'),
        ('initsegsto', 'Monitor initial volume of storage (<initsegsto>)'),
        ('segsto', 'Monitor volume of storage (<segsto>)'),
        ('levdrytosegflow', 'Monitor flow from dry cell to segment (<levdrytosegflow>)'),
        ('levmarshtosegflow', 'Monitor flow from marsh to segment (<levmarshtosegflow>)'),
        ('levdrytosegvolume', 'Monitor volume from dry cell to segment (<levdrytosegvolume>)'),
        ('levmarshtosegvolume', 'Monitor volume from marsh to segment (<levmarshtosegvolume>)'),
        ('runoff', 'Monitor volume of runoff received (<runoff>)'),
        ('wsupply', 'Monitor volume of water supply withdrawals (<wsupply>)'),
    ]

    for name, label in outputs:
        gp = ap.add_group(group_name=name, label=label)
        mdd.add_output_to_group(gp)
