"""Class for writing canal coverage data to model input files."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os.path
import xml.etree.cElementTree as Et

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import FilterLocation
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.rsm.data import monitor_data_def as mdd
from xms.rsm.file_io import util


class _MonitorData:
    """Class for holding monitor data."""
    def __init__(self, monitor_id, grp, xml_parent, monitor_xml_name):
        """Constructor.

        Args:
            monitor_id (int): monitor id
            grp (PointParameters): point data
            xml_parent (xml.etree.cElementTree.Element): xml parent element
            monitor_xml_name (str): name of the monitor xml element
        """
        self.xml_parent = xml_parent
        self.monitor_id = monitor_id
        self.monitor_xml_name = monitor_xml_name
        self.attr = grp.group_name
        self.label = grp.parameter('label').value
        self.output_type = grp.parameter('output_type').value
        self.out_file = grp.parameter('out_file').value
        if not self.out_file:
            raise RuntimeWarning('No output file specified')
        self.csv_label = grp.parameter('csv_label').value
        self.dbintl = grp.parameter('dbintl').value
        self.dss_path = grp.parameter('dss_path').value
        self.ascii_format = grp.parameter('ascii_format').value
        self.wm_id = -1
        self._monitor_xml = None
        self._check_data = {
            mdd.MONITOR_CSV: self._csv,
            mdd.MONITOR_DSS: self._dss,
            mdd.MONITOR_ASCII: self._ascii,
        }
        self._xml_data = {}

    def write(self):
        """Write the monitor data to the xml."""
        self._check_data[self.output_type]()

        xml_atts = {
            'id': str(self.monitor_id),
            'attr': self.attr,
        }
        if self.label:
            xml_atts['label'] = self.label
        if self.wm_id > 0:
            xml_atts['wmid'] = str(self.wm_id)
        monitor_xml = Et.SubElement(self.xml_parent, self.monitor_xml_name, attrib=xml_atts)
        Et.SubElement(monitor_xml, self.output_type, attrib=self._xml_data)

    def _csv(self):
        """Write the monitor data to the xml for csv."""
        if os.path.splitext(self.out_file)[1] != '.csv':
            self.out_file += '.csv'
        self._xml_data = {'file': self.out_file}
        if self.csv_label:
            self._xml_data['label'] = self.csv_label
        if self.dbintl > 0:
            self._xml_data['dbintl'] = str(self.dbintl)

    def _dss(self):
        """Write the monitor data to the xml for dss."""
        if os.path.splitext(self.out_file)[1] != '.dss':
            self.out_file += '.dss'
        self._xml_data = {'file': self.out_file}
        if self.dss_path:
            self._xml_data['pn'] = self.dss_path
        if self.dbintl > 0:
            self._xml_data['dbintl'] = str(self.dbintl)

    def _ascii(self):
        """Write the monitor data to the xml for ascii."""
        if os.path.splitext(self.out_file)[1] != '.txt':
            self.out_file += '.txt'
        self._xml_data = {
            'file': self.out_file,
        }
        if self.ascii_format:
            self._xml_data['format'] = self.ascii_format


class MonitorWriter:
    """Writer class for the monitor data."""
    def __init__(self, xml_parent, coverages):
        """Constructor.

        Args:
            xml_parent (xml.etree.cElementTree.Element): xml parent element
            coverages (list): list of coverage names
        """
        self._logger = util.get_logger()
        self._xml_parent = xml_parent
        self._coverages = coverages
        self._cov = None
        self._comp = None
        self._invalid_id = -99999
        self._feature_type = 'point'
        self._monitor_to_write = []
        self._monitor_wm_ids = []

    def _get_monitor_id(self, x: float, y: float) -> int:
        """Get a Section with parameters for a given target."""
        pass  # pragma: nocover

    def _log_invalid_monitor_id(self, pt_id, cov_name):
        """Log an invalid monitor id.

        Args:
            pt_id (int): point id
            cov_name (str): coverage name
        """
        pass  # pragma: nocover

    def _generic_model(self):
        """Get the generic model."""
        pass  # pragma: nocover

    def _get_coverage_points_with_data(self):
        """Get the coverage points with data.

        Returns:
            list: list of coverage points with data
        """
        cov_name = self._cov.name
        points = self._cov.get_points(FilterLocation.PT_LOC_DISJOINT)
        gm_pp = self._generic_model().point_parameters
        for pt in points:
            comp_id = self._comp.get_comp_id(TargetType.point, pt.id)
            if comp_id is not None and comp_id > -1:
                pt_val = self._comp.data.feature_values(TargetType.point, comp_id)
                pp = gm_pp.copy()
                pp.restore_values(pt_val)
                monitor_id = self._get_monitor_id(pt.x, pt.y)
                # save off this data to write later
                self._monitor_to_write.append((monitor_id, pp, pt.id, cov_name))

    def _write_all_monitor(self):
        """Write the monitor data to the xml."""
        if not self._monitor_to_write:
            return

        cnt = -1
        for monitor_id, pt_data, pt_id, cov_name in self._monitor_to_write:
            cnt += 1
            if monitor_id == self._invalid_id:
                self._log_invalid_monitor_id(pt_id, cov_name)
                continue
            active_groups = [grp for name, grp in pt_data.groups if grp.is_active]
            for grp in active_groups:
                try:
                    monitor_data = _MonitorData(monitor_id, grp, self._xml_parent, self._monitor_xml_name)
                    if self._monitor_wm_ids:
                        monitor_data.wm_id = self._monitor_wm_ids[cnt]
                    monitor_data.write()
                except RuntimeWarning as msg:
                    log_msg = (
                        f'{msg} for {self._feature_type} id: {pt_id} in coverage: {cov_name}.'
                        f' Monitor data was not exported for this {self._feature_type}.'
                    )
                    self._logger.warning(log_msg)

    def write(self):
        """Write the monitor data to the xml."""
        for item in self._coverages:
            self._cov = item[0]
            self._comp = item[1]
            self._get_coverage_points_with_data()
        self._write_all_monitor()
