"""Class for writing mesh properties to model input files."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import xml.etree.cElementTree as Et

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.rsm.data import mesh_data_def as mdd
from xms.rsm.file_io import util


class PropConstDatasetWriter:
    """Writer class for the rain and et control file."""
    def __init__(
        self, xms_data, property_name, option, const, const_xml_tag, const_value_xml_tag, dataset, ds_xml_tag,
        xml_parent, poly_ids_cov
    ):
        """Constructor.

        Args:
            xms_data (XmsData): Simulation data retrieved from SMS
            property_name (str): the name of the property being written
            option (str): the option to write see mesh_data_def OPT_* constants
            const (ConstValue): the constant value data class
            const_xml_tag (str): xml tag for the constant value
            const_value_xml_tag (str): xml tag for the constant value - the actual value
            dataset (Dataset): the dataset value data class
            ds_xml_tag (str): xml tag for the dataset value
            xml_parent (xml.etree.cElementTree.SubElement): mesh item in the xml
            poly_ids_cov (str): poly ids and coverage name used in error messages
        """
        self._xms_data = xms_data
        self._prop = property_name
        self._ug = self._xms_data.xmugrid
        self._option = option
        self._const = const
        self._const_xml_tag = const_xml_tag
        self._const_value_xml_tag = const_value_xml_tag
        self._dataset = dataset
        self._dataset_xml_tag = ds_xml_tag
        self._xml_parent = xml_parent
        self._poly_ids_cov = poly_ids_cov
        self._logger = util.get_logger()

    def write(self):
        """Write the rain/et portion of the control file."""
        elem = self._xml_parent
        if self._option == mdd.OPT_CONSTANT:
            atts = {self._const_value_xml_tag: f'{self._const.value}'}
            if self._const.mult != 1.0:
                atts['mult'] = f'{self._const.mult}'
            Et.SubElement(elem, self._const_xml_tag, atts)
        else:  # self._option == mdd.OPT_DATASET:
            is_ugrid_child = self._xms_data.dataset_is_child_of_simluation_ugrid(self._dataset.uuid)
            vals = self._xms_data.dset_values_from_uuid_ts(self._dataset.uuid)
            if not is_ugrid_child or len(vals) != self._ug.cell_count:
                poly_id_cov_str = self._poly_ids_cov
                msg = (
                    f'Error writing dataset for property "{self._prop}" for the following polygon(s):\n'
                    f'(polygon id, coverage name)\n{poly_id_cov_str}'
                    f'Select a valid dataset for this property.'
                )
                raise ValueError(msg)
            self._export_ds_file(self._prop, vals)
            ds_file = f'{self._prop}.dat'
            tag = self._dataset_xml_tag
            atts = {'file': ds_file}
            if self._dataset.mult != 1.0:
                atts['mult'] = f'{self._dataset.mult}'
            if self._dataset.specify_layer:
                atts['layer'] = f'{self._dataset.layer}'
            Et.SubElement(elem, tag, atts)

    def _export_ds_file(self, file_name, vals):
        """Write an XMS ascii dataset file.

        Args:
            file_name (str): file name
            vals (list[float]): list of values to write
        """
        file_name = file_name.replace(' ', '_')
        self._logger.info(f'Writing dataset to file name: {file_name}.dat')
        util.export_ds_file(file_name, vals)
