"""Class representing a mapped boundary condition coverage's component."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.gmi.components.coverage_component import CoverageComponent
from xms.guipy.dialogs.process_feedback_dlg import run_feedback_dialog

# 4. Local modules
from xms.schism.data.mapped_bc_data import MappedBcData
from xms.schism.feedback.unmap_coverage_runner import UnmapCoverageRunner
from xms.schism.gui.mapped_coverage_dialog import MappedCoverageDialog


class MappedBcComponent(CoverageComponent):
    """A hidden Dynamic Model Interface (DMI) component for the SRH-2D model simulation."""
    def __init__(self, main_file: str | Path):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        self.tree_commands.append(('View boundary conditions...', 'view_boundary_conditions'))
        self.tree_commands.append(('Convert to coverage', 'unmap_coverage'))

    def get_initial_display_options(self, query: Query, params):
        """Get initial display options."""
        main_file = Path(self.main_file)
        json_file = main_file.parent / 'arc_display.json'
        self.display_option_list = [XmsDisplayMessage(file=str(json_file), draw_type=DrawType.draw_at_locations)]
        return [], []

    def _get_data(self):
        """Get the component's data manager."""
        return MappedBcData(self.main_file)

    def unmap_coverage(self, query: Query, _params: list[dict],
                       parent: QWidget) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        Unmap the coverage.

        Args:
            query: Interprocess communication object.
            _params: Ignored.
            parent: Parent widget.

        Returns:
            Messages and requests. Always empty.
        """
        worker = UnmapCoverageRunner(self.data, query=query)
        run_feedback_dialog(worker, parent)
        return [], []

    def view_boundary_conditions(self, _query: Query, _params: list[dict],
                                 parent: QWidget) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        View mapped boundary conditions.

        Args:
            _query: Ignored.
            _params: Ignored.
            parent: Parent widget.

        Returns:
            Messages and requests. Always empty.
        """
        dialog = MappedCoverageDialog(self.data, parent)
        dialog.exec()
        return [], []
