"""Module for the MappedTidalComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query
from xms.gmi.components.gmi_component import GmiComponent

# 4. Local modules
from xms.schism.external.mapped_tidal_data import MappedTidalData
from xms.schism.external.mapped_tides_dialog import MappedTidesDialog


class MappedTidalComponent(GmiComponent):
    """A hidden Dynamic Model Interface (DMI) component for the SCHISM model simulation."""
    def __init__(self, main_file: str | Path):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(str(main_file))
        self.tree_commands.append(('Tidal constituents...', 'view_tidal_constituents'))

    def _get_data(self):
        """Get the component's data manager."""
        return MappedTidalData(self.main_file)

    def view_tidal_constituents(self, _query: Query, _params: list[dict],
                                parent: QWidget) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        View the mapped tidal constituents.

        Args:
            _query: Ignored.
            _params: Ignored.
            parent: Parent widget.

        Returns:
            Messages and requests. Always empty.
        """
        dialog = MappedTidesDialog(self.data, parent)
        dialog.exec()
        return [], []
