"""Module for the UnmapSolverCoverageRunner class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import cast

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.coverage.grid.grid_cell_to_polygon_coverage_builder import GridCellToPolygonCoverageBuilder
from xms.data_objects.parameters import Coverage
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.feedback_thread import FeedbackThread

# 4. Local modules
from xms.schism.components.coverage_component_builder import CoverageComponentBuilder
from xms.schism.components.upwind_solver_coverage_component import UpwindSolverCoverageComponent
from xms.schism.data.coverage_data import CoverageData
from xms.schism.data.mapped_upwind_solver_coverage_data import MappedUpwindSolverCoverageData
from xms.schism.data.model import get_model
from xms.schism.dmi.xms_data import XmsData
from xms.schism.external.crc import compare_crc

# 2. Third party modules


class UnmapSolverCoverageRunner(FeedbackThread):
    """Feedback thread to unmap an upwind solver coverage."""
    def __init__(self, data: MappedUpwindSolverCoverageData, query: Query):
        """Constructor."""
        super().__init__(query)
        self.data = data
        self.xms_data = XmsData(self._query)

        self.display_text = {
            'title': 'Unmapping coverage',
            'working_prompt': 'Unmapping SCHISM coverage. Please wait...',
            'warning_prompt': 'Warning(s) encountered. Review log output for more details.',
            'error_prompt': 'Error(s) encountered. Review log output for more details.',
            'success_prompt': 'Completed successfully',
            'note': '',
            'auto_load': 'Close this dialog automatically when importing is finished.'
        }

    def _run(self):
        self._log.info('Checking domain...')
        ugrid_file = self.xms_data.ugrid_file
        if not compare_crc(ugrid_file, self.data.domain_hash):
            self._log.warning('The domain was changed since the coverage was mapped. Unmapping results may be wrong.')

        self._log.info('Retrieving domain...')
        ugrid = self.xms_data.ugrid

        data = CoverageData()  # This is ugly. It's just to coax the data manager into creating a main file.
        data.close()
        component = UpwindSolverCoverageComponent(data.main_file)
        data = component.data

        name = f'{self.xms_data.current_item_name} (unmapped)'

        builder = CoverageComponentBuilder(data.main_file, name, ugrid)

        material_data = cast(list[int], self.data.solver)

        polygon_merger = GridCellToPolygonCoverageBuilder(
            co_grid=ugrid, dataset_values=material_data, projection=None, coverage_name=''
        )

        material_coverage: Coverage = polygon_merger.create_polygons_and_build_coverage()
        polygons = {polygon.id: polygon for polygon in material_coverage.polygons}

        component_ids = []
        feature_ids = []
        for material_id, polygon_ids in polygon_merger.dataset_polygon_ids.items():
            section = get_model().material_parameters
            if material_id == 0:
                group = 'upwind'
            else:
                group = 'higher_order'

            section.group(group).enabled = True
            values = section.extract_values()

            for polygon_id in polygon_ids:
                component_id = data.add_feature(TargetType.polygon, values, group)
                component_ids.append(component_id)
                feature_ids.append(polygon_id)
                builder.add_polygon(polygons[polygon_id], component_id)

        coverage = builder.complete()

        component.cov_uuid = coverage.uuid
        data.commit()
        for component_id in component_ids:
            component.update_component_id(TargetType.arc, component_id, component_id)
        component.update_id_files()

        self.xms_data.add_coverage(coverage, component, upwind_solver=True)

        self._log.info('The coverage was unmapped')
