"""Module for the VGrid writer."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.schism.file_io.vgrid.vgrid_file import VGridFile


def write_vgrid(vgrid: VGridFile, path: Path | str):
    """
    Write a VGrid to disk.

    Args:
        vgrid: The VGrid to write.
        path: Where to write it.
    """
    with VGridWriter(vgrid, path) as writer:
        writer.write()


class VGridWriter:
    """Class for writing SCHISM VGrids."""
    def __init__(self, vgrid: VGridFile, path: Path):
        """
        Initialize the writer.

        Args:
            vgrid: The VGrid to write.
            path: Where to write the grid to.
        """
        self._vgrid = vgrid
        self._path = path

    def __enter__(self):
        """Open the file."""
        self._line_number = 0
        self._file = open(self._path, 'w')
        return self

    def __exit__(self, _exc_type, _exc_value, _exc_tb):
        """Close the file."""
        self._file.close()

    def write(self):
        """Write the VGrid to disk."""
        vgrid = self._vgrid

        self._file.write('2 !ivcor\n')

        nvrt = len(vgrid.z_levels) + len(vgrid.s_levels) - 1  # one of the levels is shared
        kz = len(vgrid.z_levels)
        hs = vgrid.z_levels[-1]
        self._file.write(f'{nvrt} {kz} {hs} !nvrt, kz (# of Z-levels); h_s (transition depth between S and Z)\n')

        self._file.write('Z levels\n')
        for i, z in enumerate(vgrid.z_levels, start=1):
            self._file.write(f'{i} {-z}\n')  # invert z, depth -> elevation

        self._file.write('S levels\n')
        self._file.write(f'{vgrid.hc} {vgrid.theta_b} {vgrid.theta_f} !h_c, theta_b, theta_f\n')

        for i, s in enumerate(vgrid.s_levels, start=1):
            self._file.write(f'{i} {-s}\n')  # invert s, depth -> elevation
