#! python3
# import uuid
from geopandas import GeoDataFrame
import logging
from pathlib import Path
from typing import Callable

from ._xmssnap.snap import _SnapPolygon
from .snap_base import _SnapBase, SnapBase
from xms.constraint import read_grid_from_file
from xms.data_objects.parameters import FilterLocation, Polygon, UGrid


#: The "feature ID" assigned to cells that don't have a polygon of their own.
NO_POLYGON = -1


def _get_points_from_arc_id(cov: GeoDataFrame, arc_id: int) -> list | None:
    arcs = cov[cov['geometry_types'] == 'Arc']
    for arc in arcs.itertuples():
        if arc.id == arc_id:
            return list(arc.geometry.coords)


class SnapPolygon(_SnapPolygon, SnapBase):
    """
    This class snaps arcs to the boundary of a geometry.

    The present algorithm boils down to finding which polygon the cell's centroid is in or on and assigning the cell to
    that polygon. Centroids that aren't in any polygon will be assigned to NO_POLYGON above. Centroids on the boundary
    between a polygon and a hole/the exterior are preferentially assigned to the polygon. Centroids that are in/on
    multiple polygons will be assigned to one of the polygons arbitrarily. The particular tie-breaker algorithm is an
    implementation detail and subject to change at a future date, so relying on it for tests is probably a bad idea.
    """
    def __init__(self):
        """Constructor."""
        _SnapPolygon.__init__(self)
        SnapBase.__init__(self)
        self.progress_callback = logging.getLogger('xms.snap').info
        self.set_progress_callback(self.progress_callback)

    def set_grid(self, grid, target_cells):
        """Sets the geometry that will be snapped to.

        Args:
            grid (data_objects.parameters.UGrid): The grid that will be targeted.
            target_cells (bool): True if the snap targets cell centers, point locations if false.
        """
        if isinstance(grid, UGrid):  # data_objects UGrid
            file = grid.cogrid_file
            if file:  # New CoGrid impl
                co_grid = read_grid_from_file(file)
            else:  # Old C++ impl for H5 file format
                co_grid = super().get_co_grid(grid)
            _SnapBase.set_grid(self, co_grid._instance, target_cells)
        else:
            _SnapBase.set_grid(self, grid._instance, target_cells)

    def add_polygons(self, polygons: list[Polygon], test_dump_stem: str | Path = ''):
        """
        Add polygons to the snapper.

        This also snaps the polygons too due to historical reasons (that's how it was originally made, and now everyone
        depends on it). This operation may be slow on large UGrids and complex coverages.

        Polygons should have unique feature IDs that are all greater than zero. Due to the deconstruction/reconstruction
        used to get polygons across the language boundary to the snapper, polygons with duplicate IDs may be corrupted
        and cause bizarre results. Polygons with IDs <1 may be mishandled by the snapper and cause unpleasant surprises.

        Args:
            polygons: List of polygons to add.
            test_dump_stem: A tool for debugging the snapper. If nonempty, files named <test_dump_stem>.xmc and
                <test_dump_stem>.txt will be written, which together contain a serialized form of the snapper in the
                state immediately before snapping begins. A test can then be written on the C++ side which passes the
                same stem to SnapPolygon::LoadTestDump to recreate the same snapper. This makes it possible to debug an
                equivalent snapper without having to cross the boundary between Python and C++.
        """
        self.progress_callback('Deconstructing polygons...')
        polygon_ids = []
        poly_idxs = []
        arcs = []
        arc_directions = []

        for poly in polygons:
            polygon_id = poly.id
            poly_idx = 0

            # add outer polygon arcs
            poly_arcs = poly.arcs
            dirs = poly.arc_directions
            for i in range(len(poly_arcs)):
                pts = poly_arcs[i].get_points(FilterLocation.LOC_ALL)
                arc_pts = [(pt.x, pt.y, pt.z) for pt in pts]
                direction = True if dirs[i] else False
                polygon_ids.append(polygon_id)
                poly_idxs.append(poly_idx)
                arcs.append(arc_pts)
                arc_directions.append(direction)
            poly_idx += 1

            # add inner polygon arcs
            holes = poly.interior_arcs
            hole_dirs = poly.interior_arc_directions
            for i in range(len(holes)):
                for j in range(len(holes[i])):
                    pts = holes[i][j].get_points(FilterLocation.LOC_ALL)
                    arc_pts = [(pt.x, pt.y, pt.z) for pt in pts]
                    direction = True if hole_dirs[i][j] else False
                    polygon_ids.append(polygon_id)
                    poly_idxs.append(poly_idx)
                    arcs.append(arc_pts)
                    arc_directions.append(direction)
                poly_idx += 1
        test_dump_stem = str(test_dump_stem)
        self.progress_callback('Adding polygons...')
        super().add_polygons(polygon_ids, poly_idxs, arcs, arc_directions, test_dump_stem)
        self.snap_polygons()

    def add_gdf_polygons(self, cov: GeoDataFrame, test_dump_stem: str | Path = ''):
        """
        Add polygons to the snapper.

        This also snaps the polygons too due to historical reasons (that's how it was originally made, and now everyone
        depends on it). This operation may be slow on large UGrids and complex coverages.

        Polygons should have unique feature IDs that are all greater than zero. Due to the deconstruction/reconstruction
        used to get polygons across the language boundary to the snapper, polygons with duplicate IDs may be corrupted
        and cause bizarre results. Polygons with IDs <1 may be mishandled by the snapper and cause unpleasant surprises.

        Args:
            cov: coverage GeoDataFrame with polygons to add.
            test_dump_stem: A tool for debugging the snapper. If nonempty, files named <test_dump_stem>.xmc and
                <test_dump_stem>.txt will be written, which together contain a serialized form of the snapper in the
                state immediately before snapping begins. A test can then be written on the C++ side which passes the
                same stem to SnapPolygon::LoadTestDump to recreate the same snapper. This makes it possible to debug an
                equivalent snapper without having to cross the boundary between Python and C++.
        """
        self.progress_callback('Deconstructing polygons...')
        polygon_ids = []
        poly_idxs = []
        arcs = []
        arc_directions = []

        polygons = cov[cov['geometry_types'] == 'Polygon']
        for poly in polygons.itertuples():
            polygon_id = poly.id
            poly_idx = 0

            # add outer polygon arcs
            for arc_id, arc_dir in zip(poly.polygon_arc_ids, poly.polygon_arc_directions):
                arc_pts = _get_points_from_arc_id(cov, arc_id)
                direction = True if arc_dir else False
                polygon_ids.append(polygon_id)
                poly_idxs.append(poly_idx)
                arcs.append(arc_pts)
                arc_directions.append(direction)
            poly_idx += 1

            # add inner polygon arcs
            for poly_arc_ids, poly_directions in zip(poly.interior_arc_ids, poly.interior_arc_directions):
                for arc_id, arc_dir in zip(poly_arc_ids, poly_directions):
                    arc_pts = _get_points_from_arc_id(cov, arc_id)
                    direction = True if arc_dir else False
                    polygon_ids.append(polygon_id)
                    poly_idxs.append(poly_idx)
                    arcs.append(arc_pts)
                    arc_directions.append(direction)
                poly_idx += 1
        test_dump_stem = str(test_dump_stem)
        self.progress_callback('Adding polygons...')
        super().add_polygons(polygon_ids, poly_idxs, arcs, arc_directions, test_dump_stem)
        self.snap_polygons()

    def get_cells_in_polygon(self, polygon: int) -> tuple[int]:
        """
        Get the cell indexes that were snapped to a polygon.

        Args:
            polygon: The feature ID of the polygon to get cells for.

        Returns:
            The indexes (in the UGrid passed to self.set_grid) of each cell that was snapped to the requested polygon.
            If no cells were snapped to it, or the polygon wasn't even passed in to self.set_polygons, then an empty
            sequence is returned.
        """
        # This is just a pass-through so there's documentation of it where developers expect.
        return super().get_cells_in_polygon(polygon)

    def get_cell_to_polygon(self) -> tuple[int]:
        """
        Get a mapping from cell_index->feature_id.

        The sequence is parallel to the list of cells in the UGrid passed to self.set_ugrid, and each element is the
        feature ID of the polygon the cell was assigned to. Cells that were not assigned to any polygon will be
        assigned to the NO_POLYGON constant from above.

        Returns:
            The mapping.
        """
        # This is just a pass-through so there's documentation of it where developers expect.
        return super().get_cell_to_polygon()

    def set_progress_callback(self, callback: Callable[[str], None]):
        """
        Set a callback used to report progress.

        Args:
            callback: The callback that should be called when progress updates are available.
        """
        # This is just a pass-through so there's documentation of it where developers expect.
        super().set_progress_callback(callback)
