"""ModelControlAdvanced class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler

# 4. Local modules


class ModelControlAdvanced(param.Parameterized):
    """Manages data file for the hidden simulation component."""
    turbulence_model = param.ObjectSelector(
        default='Parabolic',
        objects=['Laminar', 'Constant', 'Parabolic', 'KE'],
        doc='turbulence model used in the simulation',
        precedence=1,
    )
    parabolic_turbulence = param.Number(
        default=0.7,
        bounds=(0.01, 1.0),
        doc='parabolic turbulence',
        precedence=1.1,
    )
    unsteady_output = param.Boolean(
        default=True,
        doc='unsteady flow',
        precedence=2,
    )
    run_in_partial_mode = param.Boolean(
        label='Run SRH-Pre in partial mode',
        default=False,
        doc='run SRH2D in partial mode using the SIF as input',
        precedence=4,
    )
    use_secondary_time_step = param.Boolean(
        label='Use initial and secondary time step',
        default=False,
        doc='specify a secondary time step that will become active during the simulation',
        precedence=5,
    )
    secondary_time_step = param.Number(
        default=1.0,
        bounds=(0.0001, None),
        doc='secondary time step',
        precedence=5.1,
    )
    activation_time = param.Number(
        label='Activation time for secondary time step',
        default=0.0,
        bounds=(0.0, None),
        doc='activation time',
        precedence=5.2,
    )
    use_termination_criteria = param.Boolean(
        default=False,
        doc='use termination criteria',
        precedence=6,
    )
    netq_value = param.Number(
        label='Stop simulation when Net_Q/INLET_Q is less than',
        default=0.01,
        bounds=(0.000001, .25),
        doc='NetQ termination value',
        precedence=6.1,
    )
    specify_materials_nlcd = param.Boolean(
        default=False,
        label='Specify materials using NLCD raster',
        doc='Specify materials using NLCD raster',
        precedence=7,
    )
    select_nlcd_raster = param.Action(
        doc='Select NLCD raster.',
        label='Select NLCD raster...',
        precedence=7.1,
    )
    raster_label = param.String(
        doc='Name of NLCD raster.',
        default='No raster selected',
        label='',
        precedence=7.2,
    )
    raster_uuid = param.String(
        doc='UUID of NLCD raster.',
        default='',
        label='',
        precedence=-7.3,
    )
    background_manning_n = param.Number(
        label="Background Manning's N",
        default=0.03,
        bounds=(0.0, None),
        doc='Background Manning N',
        precedence=7.4,
    )
    # keep these 2 variables around for old IO, but they are not used
    specify_pressure_dataset = param.Boolean(
        default=False,
        doc='',
        precedence=-3,
    )
    pressure_dataset = param.String(
        default='No Dataset selected',
        doc='UUID of dataset selected from xms',  # Time step number (1-based) may be appended with a colon ('uuid:ts')
        precedence=-3.1,
    )

    def __init__(self):
        """Initializes the data class.

        Args:
            data_file (:obj:`str`): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.

        """
        super().__init__()
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=[
                'turbulence_model', 'use_secondary_time_step', 'use_termination_criteria', 'specify_materials_nlcd'
            ],
            depends={
                'Parabolic': ['parabolic_turbulence'],
                'use_secondary_time_step': ['secondary_time_step', 'activation_time'],
                'use_termination_criteria': ['netq_value'],
                'specify_materials_nlcd': ['select_nlcd_raster', 'raster_label', 'background_manning_n'],
            }
        )

    @param.depends('turbulence_model', watch=True)
    def _update_turbulence_model(self):
        """Watches changes made to the turbulence_model object."""
        self.enabler.do_enabling(force_disable=False)

    @param.depends('use_secondary_time_step', watch=True)
    def _update_use_secondary_time_step(self):
        """Watches changes made to the use_secondary_time_step object."""
        self.enabler.do_enabling(force_disable=False)

    @param.depends('use_termination_criteria', watch=True)
    def _update_use_termination_criteria(self):
        """Watches changes made to the use_termination_criteria object."""
        self.enabler.do_enabling(force_disable=False)

    @param.depends('specify_materials_nlcd', watch=True)
    def _update_specify_materials_nlcd(self):
        """Watches changes made to the use_termination_criteria object."""
        self.enabler.do_enabling(force_disable=False)
