"""ModelControlOutput class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler

# 4. Local modules


class ModelControlOutput(param.Parameterized):
    """Manages data file for the hidden simulation component."""
    output_format = param.ObjectSelector(
        default='XMDF',
        objects=['XMDF', 'SRHN', 'SRHC', 'TECPlot', 'Paraview'],
        doc='output dataset file format',
        precedence=1,
    )
    output_units = param.ObjectSelector(
        default='English',
        objects=['English', 'Metric'],
        doc='output dataset units',
        precedence=2,
    )
    output_method = param.ObjectSelector(
        default='Specified Frequency',
        objects=['Specified Frequency', 'Specified Times', 'Simulation End'],
        doc='option for output times',
        precedence=3,
    )
    output_frequency = param.Number(
        default=5.0,
        bounds=(0.0, None),
        doc='frequency that output is written to output files',
        precedence=3.1,
    )
    output_frequency_units = param.ObjectSelector(
        label='',
        default='Minutes',
        objects=['Days', 'Hours', 'Minutes', 'Seconds'],
        doc='units for output frequency',
        precedence=3.2,
    )
    output_specified_times = param.DataFrame(
        default=pd.DataFrame(data={'Times': [0.0]}),
        doc='list of times for output',
        precedence=3.3,
    )
    output_maximum_datasets = param.Boolean(
        default=False,
        doc='enable output of max value datasets',
        precedence=4,
    )
    specify_maximum_dataset_time_range = param.Boolean(
        default=False,
        doc='specify a time range on the max dataset output',
        precedence=4.1,
    )
    maximum_dataset_minimum_time = param.Number(
        default=0.0,
        bounds=(0.0, None),
        doc='minimum time for max dataset output',
        precedence=4.2,
    )
    maximum_dataset_maximum_time = param.Number(
        default=1.0,
        bounds=(0.0, None),
        doc='maximum time for max dataset output',
        precedence=4.3,
    )

    def __init__(self):
        """Initializes the data class.

        Args:
            data_file (:obj:`str`): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.

        """
        super().__init__()
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=['output_method', 'output_maximum_datasets', 'specify_maximum_dataset_time_range'],
            depends={
                'Specified Frequency': ['output_frequency', 'output_frequency_units'],
                'Specified Times': ['output_specified_times'],
                'output_maximum_datasets': ['specify_maximum_dataset_time_range'],
                'specify_maximum_dataset_time_range': ['maximum_dataset_minimum_time', 'maximum_dataset_maximum_time'],
            }
        )
        self.enabler.do_enabling(force_disable=False)

    @param.depends('output_method', watch=True)
    def _update_output_method(self):
        """Watches changes made to the output_method object."""
        self.enabler.do_enabling(force_disable=False)

    @param.depends('output_maximum_datasets', watch=True)
    def _update_output_maximum_datasets(self):
        """Watches changes made to the output_maximum_datasets object."""
        self.enabler.do_enabling(force_disable=False)
        if not self.output_maximum_datasets:
            self.specify_maximum_dataset_time_range = False

    @param.depends('specify_maximum_dataset_time_range', watch=True)
    def _update_specify_maximum_dataset_time_range(self):
        """Watches changes made to the specify_maximum_dataset_time_range object."""
        self.enabler.do_enabling(force_disable=False)
