"""Reads SRH-2D material file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import shlex
import sys

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class MaterialReader:
    """Reads SRH-2D material file."""
    def __init__(self, is_sediment):
        """Constructor.

        Args:
            is_sediment (:obj:`bool`): True if the file contains sediment materials
        """
        self.logger = logging.getLogger('xms.srh')
        self.material_count = 0
        self.material_names = {}  # Dict of material id to material names
        self.material_cells = {}  # Dict of material id and list of cells
        self.lines = []  # List of lines from the file
        self.curr_line = 0  # Current line in the file we're reading
        self.line_count = 0  # Number of lines in the file
        if is_sediment:
            self._first_line_identifier = 'SRHSEDMAT 30'
            self._num_mats_card = 'nsedmaterials'
            self._mat_name_card = 'sedmatname'
            self._mat_cells_card = 'sedmaterial'
        else:
            self._first_line_identifier = 'SRHMAT 30'
            self._num_mats_card = 'nmaterials'
            self._mat_name_card = 'matname'
            self._mat_cells_card = 'material'

    def _read_file_type(self):
        """Reads file type on first line of file and raises error if incorrect."""
        if self.lines[0] != self._first_line_identifier:
            raise RuntimeError('Material file format is not recognized.')
        self.curr_line += 1

    def _read_material_count(self):
        """Reads the number of materials."""
        if self.curr_line < self.line_count:
            words = self.lines[self.curr_line].split()
            if words and len(words) > 1:
                self.material_count = int(words[1])
        self.curr_line += 1

    def _read_material_names(self):
        """Reads the material names."""
        while self.curr_line < self.line_count:
            if self.lines[self.curr_line].lower().startswith(self._mat_name_card):
                # Use shlex to handle quoted strings
                words = shlex.split(self.lines[self.curr_line], posix="win" not in sys.platform)
                if words and len(words) > 2:
                    mat_id = int(words[1])
                    self.material_names[mat_id] = words[2].strip('"\'')
                    if len(self.material_names) == self.material_count - 1:
                        break
            self.curr_line += 1

        self.material_names[0] = 'unassigned'

        self.curr_line += 1

    def _read_cells_for_material(self, words):
        """Reads the cells associated with the material.

        The list of cells may wrap on several lines. Keep reading lines until the first word is not an integer.

        Args:
            words: List of strings from the first line of the material minus 'Material' and the material number.

        Returns:
            (:obj:`list[int]`: Cell ids associated with the material.
        """
        cells = []
        while words:
            if not words[0].isdigit():
                self.curr_line -= 1
                break
            cells.extend([(int(i) - 1) for i in words])
            self.curr_line += 1
            if self.curr_line >= self.line_count:
                break
            words = self.lines[self.curr_line].split()

        return cells

    def _read_material_cells(self):
        """Reads the cells assigned to the materials for all materials."""
        # Make sure the unassigned material is in the dict
        self.material_cells[0] = []

        self.logger.info('Reading material cells.')
        words_to_skip = 2  # Skip over the 'Material' word and the material number
        while self.curr_line < self.line_count:
            words = self.lines[self.curr_line].split()
            if words and words[0].lower() == self._mat_cells_card:
                if len(words) > 1:
                    material_id = int(words[1])
                    cells = self._read_cells_for_material(words[words_to_skip:])
                    self.material_cells[material_id] = cells
            self.curr_line += 1

    def read(self, filename):
        """Reads the material file.

        Args:
            filename (:obj:`str`): Filepath of material file.
        """
        try:
            with open(filename, 'r') as file:
                self.lines = file.read().splitlines()
                self.line_count = len(self.lines)
                self._read_file_type()
                self._read_material_count()
                self._read_material_names()
                self._read_material_cells()

        except Exception as error:  # pragma: no cover
            self.logger.exception(f'Error reading materials: {str(error)}')
            raise error
