"""Writes SRH-2D material and sediment material files."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class MaterialWriter:
    """Writes SRH-2D geom file."""
    def __init__(self, file_name, mat_names, mat_grid_cells):
        """Constructor.

        Args:
            file_name (:obj:`str`): Path to the output file
            mat_names (:obj:`list[str]`): Names of the materials in the coverage's material list
            mat_grid_cells (:obj:`dict`): The grid cell material polygon assignments. Key = material_id,
                Value = [cell_id]
        """
        self._file_name = file_name
        self._mat_names = mat_names
        self._mat_grid_cells = mat_grid_cells
        is_sediment = file_name.lower().endswith('srhsedmat')
        if is_sediment:
            self._first_line_identifier = 'SRHSEDMAT 30'
            self._num_mats_card = 'NSedMaterials'
            self._mat_name_card = 'SedMatName'
            self._mat_cells_card = 'SedMaterial'
        else:
            self._first_line_identifier = 'SRHMAT 30'
            self._num_mats_card = 'NMaterials'
            self._mat_name_card = 'MatName'
            self._mat_cells_card = 'Material'

    def write(self):
        """Write the SRH-2D materials file."""
        with open(self._file_name, 'w') as file:
            file.write(f'{self._first_line_identifier}\n')
            file.write(f'{self._num_mats_card} {len(self._mat_names)}\n')
            for i in range(1, len(self._mat_names)):
                file.write(f'{self._mat_name_card} {i} "{self._mat_names[i]}"\n')
            for mat_id, cells in self._mat_grid_cells.items():
                if mat_id == 0 or len(cells) < 1:
                    continue  # Skip the unassigned material
                file.write(f'{self._mat_cells_card} {mat_id}')
                for j in range(len(cells)):
                    file.write(f' {cells[j] + 1}')
                    if (j + 1) % 10 == 0:
                        file.write('\n')
                if len(cells) % 10 != 0:
                    file.write('\n')
